package ru.yandex.direct.grid.core.entity.showcondition.service;

import java.time.LocalDate;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowCondition;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiAggregatedShowConditionOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridAggregatedShowConditionYtRepository;

import static ru.yandex.direct.multitype.entity.LimitOffset.limited;

/**
 * Сервис для получения аггрегированной информации об условиях показа клиентов
 */
@Service
@ParametersAreNonnullByDefault
public class GridAggregatedShowConditionService {
    public static final Integer MAX_CONDITION_ROWS = 10_000;

    private final FeatureService featureService;
    private final MetrikaGoalsService metrikaGoalsService;
    private final GridAggregatedShowConditionYtRepository gridAggregatedShowConditionYtRepository;

    @Autowired
    public GridAggregatedShowConditionService(
            FeatureService featureService,
            MetrikaGoalsService metrikaGoalsService,
            GridAggregatedShowConditionYtRepository gridAggregatedShowConditionYtRepository) {
        this.featureService = featureService;
        this.metrikaGoalsService = metrikaGoalsService;
        this.gridAggregatedShowConditionYtRepository = gridAggregatedShowConditionYtRepository;
    }

    /**
     * Получить данные об условиях показа
     *
     * @param shard                              шард, в котором хранятся условия показа (исключительно для улучшения результатов запроса)
     * @param operatorUid                        uid оператора
     * @param clientId                           id клиента
     * @param filter                             настройки фильтрации выбранных условий показа
     * @param orderBy                            настройки упорядочивания условий показа
     * @param statStartDay                       начало периода, за который мы получаем статистику по условиям показа
     * @param statEndDay                         конец периода, за который мы получаем статистику по условиям показа
     * @param goalIds                            идентификаторы целей
     */
    public List<GdiAggregatedShowCondition> getShowConditions(int shard, Long operatorUid, ClientId clientId,
                                                              GdiAggregatedShowConditionFilter filter,
                                                              List<GdiAggregatedShowConditionOrderBy> orderBy,
                                                              LocalDate statStartDay, LocalDate statEndDay,
                                                              Set<Long> goalIds) {
        boolean getRevenueOnlyByAvailableGoals =
                featureService.isEnabledForClientId(clientId, FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
        Set<Long> availableGoalIds = null;
        if (CollectionUtils.isNotEmpty(goalIds) && getRevenueOnlyByAvailableGoals) {
            availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                    operatorUid, clientId);
        }
        return gridAggregatedShowConditionYtRepository
                .getAggregatedShowConditions(shard, filter, orderBy, statStartDay, statEndDay,
                        limited(MAX_CONDITION_ROWS), goalIds, availableGoalIds);
    }
}
