package ru.yandex.direct.grid.core.entity.showcondition.service;

import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableList;
import one.util.streamex.StreamEx;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.model.Goal;
import ru.yandex.direct.core.entity.retargeting.model.GoalBase;
import ru.yandex.direct.core.entity.retargeting.model.GoalType;
import ru.yandex.direct.core.entity.retargeting.model.Retargeting;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingGoalsRepository;
import ru.yandex.direct.core.entity.retargeting.repository.RetargetingRepository;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingConditionService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingCondition;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingConditionRuleItem;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridRetargetingYtRepository;
import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceProfile;

import static ru.yandex.direct.utils.FunctionalUtils.listToMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;

@Service
@ParametersAreNonnullByDefault
public class GridRetargetingConditionService {
    public static final List<String> GEO_SEGMENT_SUBTYPES =
            ImmutableList.of("geo_last", "geo_regular", "geo_condition");
    public static final String DEFAULT_AUDIENCE_DOMAIN = "audience.yandex.ru";

    private final RetargetingGoalsRepository retargetingGoalsRepository;

    private final RetargetingConditionService retargetingConditionService;

    private final GridRetargetingYtRepository retargetingYtRepository;
    private final RetargetingRepository retargetingRepository;

    @Autowired
    public GridRetargetingConditionService(
            RetargetingGoalsRepository retargetingGoalsRepository,
            RetargetingConditionService retargetingConditionService,
            GridRetargetingYtRepository retargetingYtRepository,
            RetargetingRepository retargetingRepository) {
        this.retargetingGoalsRepository = retargetingGoalsRepository;
        this.retargetingConditionService = retargetingConditionService;
        this.retargetingYtRepository = retargetingYtRepository;
        this.retargetingRepository = retargetingRepository;
    }

    /**
     * Условия ретаргетинга для клиента
     *
     * @param clientId идентификатор клиента
     * @param filter   фильтр условий ретаргетинга
     */
    public List<GdiRetargetingCondition> getRetargetingConditions(int shard, ClientId clientId,
                                                                  GdiRetargetingConditionFilter filter) {
        List<GdiRetargetingCondition> retargetingConditions;
        try (TraceProfile ignore =
                     Trace.current().profile("GridRetargetingConditionService:getRetargetingConditions")) {
            retargetingConditions = retargetingYtRepository.getRetargetingConditions(shard, filter);
        }
        Set<Long> retargetingConditionIds = listToSet(retargetingConditions,
                GdiRetargetingCondition::getRetargetingConditionId);

        List<Retargeting> retargetings;
        try (TraceProfile ignore =
                     Trace.current().profile("GridRetargetingConditionService:getRetargetingsByRetCondIds")) {
            retargetings = retargetingRepository.getRetargetingsByRetCondIds(shard, retargetingConditionIds);
        }
        Map<Long, List<Retargeting>> retargetingsByRetCondId = StreamEx.of(retargetings)
                .groupingBy(Retargeting::getRetargetingConditionId);

        Set<Long> accessibleRetargetingConditions =
                retargetingGoalsRepository.getAccessibleRetConditionIds(shard, retargetingConditionIds);

        var retargetingGoalIds = StreamEx.of(retargetingConditions)
                .flatCollection(GdiRetargetingCondition::getConditionRules)
                .flatCollection(GdiRetargetingConditionRuleItem::getGoals)
                .map(GoalBase::getId)
                .toSet();
        List<Goal> availableMetrikaGoalsForRetargeting =
                retargetingConditionService.getAvailableMetrikaGoalsForRetargeting(clientId, retargetingGoalIds);
        Map<Long, Goal> availableMetrikaGoalsById = listToMap(availableMetrikaGoalsForRetargeting, Goal::getId);
        Map<Long, Goal> metrikaGoalsById = getMetrikaGoalsForRetargeting(clientId, availableMetrikaGoalsForRetargeting);

        retargetingConditions.forEach(
                condition -> {
                    Long retCondId = condition.getRetargetingConditionId();
                    List<Retargeting> retargetingList = retargetingsByRetCondId.getOrDefault(retCondId,
                            Collections.emptyList());

                    condition
                            .withCampaignIds(listToSet(retargetingList, Retargeting::getCampaignId))
                            .withAdGroupIds(listToSet(retargetingList, Retargeting::getAdGroupId))
                            .withHasUnavailableGoals(!accessibleRetargetingConditions.contains(retCondId));
                    enrichRetargetingConditionWithGoalInfo(condition, availableMetrikaGoalsById, metrikaGoalsById);
                });

        return retargetingConditions;
    }

    private Map<Long, Goal> getMetrikaGoalsForRetargeting(ClientId clientId,
                                                          List<Goal> availableMetrikaGoalsForRetargeting) {
        List<Goal> metrikaGoalsForRetargeting = retargetingConditionService.getMetrikaGoalsForRetargeting(
                clientId, availableMetrikaGoalsForRetargeting);
        return listToMap(metrikaGoalsForRetargeting, Goal::getId);
    }

    private static void enrichRetargetingConditionWithGoalInfo(GdiRetargetingCondition retargetingCondition,
                                                               Map<Long, Goal> availableMetrikaGoalsById,
                                                               Map<Long, Goal> metrikaGoalsById) {
        Set<Goal> retargetingConditionGoalsInfo = StreamEx.of(retargetingCondition.getConditionRules())
                .map(GdiRetargetingConditionRuleItem::getGoals)
                .flatMap(Collection::stream)
                .map(Goal::getId)
                .filter(availableMetrikaGoalsById::containsKey)
                .map(availableMetrikaGoalsById::get)
                .toSet();

        // добавим значение name для цели, поскольку оно не читается из yt (там этого поля нет)
        for (var conditionRule : retargetingCondition.getConditionRules()) {
            for (var goal : conditionRule.getGoals()) {
                var goalData = metrikaGoalsById.get(goal.getId());
                if (goalData == null) {
                    continue;
                }

                goal.setName(goalData.getName());
                // для lal-сегментов добавляем тип родительской цели
                if (goal.getType() == GoalType.LAL_SEGMENT) {
                    goal.setLalParentType(goalData.getLalParentType());
                }
            }
        }

        retargetingConditionGoalsInfo.stream()
                .filter(goal -> goal.getType() == GoalType.AUDIENCE)
                .forEach(x -> x.setDomain(DEFAULT_AUDIENCE_DOMAIN));

        Set<Long> goalIds = listToSet(retargetingConditionGoalsInfo, Goal::getId);

        Set<String> goalDomains = StreamEx.of(retargetingConditionGoalsInfo)
                .map(Goal::getDomain)
                .nonNull()
                .remove(String::isEmpty)
                .toSet();

        retargetingCondition
                .withGoalIds(goalIds)
                .withGoalDomains(goalDomains)
                .withHasGeoSegments(hasGeoSegmentSubtype(retargetingConditionGoalsInfo));
    }

    private static boolean hasGeoSegmentSubtype(Collection<Goal> goals) {
        return goals.stream()
                .map(Goal::getSubtype)
                .anyMatch(GEO_SEGMENT_SUBTYPES::contains);
    }

}
