package ru.yandex.direct.grid.core.entity.showcondition.service;

import java.time.LocalDate;
import java.util.List;
import java.util.Map;

import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.retargeting.container.RetargetingSelection;
import ru.yandex.direct.core.entity.retargeting.model.TargetInterest;
import ru.yandex.direct.core.entity.retargeting.service.RetargetingService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.RetargetingFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingFilter;
import ru.yandex.direct.grid.core.entity.retargeting.model.GdiRetargetingOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiBidsRetargetingWithTotals;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridRetargetingYtRepository;
import ru.yandex.direct.multitype.entity.LimitOffset;

import static java.util.function.Function.identity;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

@Service
@ParametersAreNonnullByDefault
public class GridRetargetingService {
    private final ShardHelper shardHelper;

    private final RetargetingService retargetingService;

    private final GridRetargetingYtRepository retargetingYtRepository;

    @Autowired
    public GridRetargetingService(RetargetingService retargetingService,
                                  GridRetargetingYtRepository retargetingYtRepository,
                                  ShardHelper shardHelper) {
        this.retargetingService = retargetingService;
        this.retargetingYtRepository = retargetingYtRepository;
        this.shardHelper = shardHelper;
    }

    /**
     * Проверяет есть ли в группе клиента ретаргетинги
     *
     * @return для каждого id группы возвращает есть ли в ней ретаргетинг
     */
    public Map<Long, Boolean> getContainingOfRetargetingsForAdGroupIds(ClientId clientId, Long operatorUid,
                                                                       List<Long> adGroupIds) {
        RetargetingSelection retargetingSelection = new RetargetingSelection().withAdGroupIds(adGroupIds);

        List<TargetInterest> retargetings =
                retargetingService
                        .getRetargetings(retargetingSelection, clientId, operatorUid, LimitOffset.maxLimited());

        Map<Long, TargetInterest> retargetingByAdGroupId = listToMap(retargetings, TargetInterest::getAdGroupId);

        return listToMap(adGroupIds, identity(), retargetingByAdGroupId::containsKey);
    }

    /**
     * Ставки ретаргетинга клиента
     *
     * @param disableStatusFilter  не фильтровать по статусам
     * @param addWithTotalsToQuery добавляет в запросе with totals с получением общей сатистики
     */
    public GdiBidsRetargetingWithTotals getRetargetings(ClientId clientId, GdiRetargetingFilter filter,
                                                        RetargetingFetchedFieldsResolver fetchingFieldsResolver,
                                                        LocalDate statFrom, LocalDate statTo,
                                                        List<GdiRetargetingOrderBy> orderByList,
                                                        boolean disableStatusFilter, boolean addWithTotalsToQuery) {
        int shard = shardHelper.getShardByClientIdStrictly(clientId);

        return retargetingYtRepository.getRetargetings(shard, clientId, filter,
                fetchingFieldsResolver, orderByList, statFrom, statTo,
                limited(GridShowConditionConstants.getMaxConditionRows()),
                disableStatusFilter, addWithTotalsToQuery);
    }
}
