package ru.yandex.direct.grid.core.entity.showcondition.service;

import java.time.LocalDate;
import java.util.List;
import java.util.Set;

import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.ShowConditionFetchedFieldsResolver;
import ru.yandex.direct.grid.core.entity.group.service.GridAdGroupConstants;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionFilter;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionOrderBy;
import ru.yandex.direct.grid.core.entity.showcondition.model.GdiShowConditionWithTotals;
import ru.yandex.direct.grid.core.entity.showcondition.repository.GridShowConditionYtRepository;

import static com.google.common.base.Preconditions.checkState;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.multitype.entity.LimitOffset.limited;

/**
 * Сервис для получения базовой информации об условиях показа клиентов
 */
@Service
@ParametersAreNonnullByDefault
public class GridShowConditionService {
    private final FeatureService featureService;
    private final MetrikaGoalsService metrikaGoalsService;
    private final GridShowConditionYtRepository gridShowConditionYtRepository;

    @Autowired
    public GridShowConditionService(FeatureService featureService,
                                    MetrikaGoalsService metrikaGoalsService,
                                    GridShowConditionYtRepository gridShowConditionYtRepository) {
        this.featureService = featureService;
        this.metrikaGoalsService = metrikaGoalsService;
        this.gridShowConditionYtRepository = gridShowConditionYtRepository;
    }

    /**
     * Получить данные об условиях показа
     *
     * @param shard                              шард, в котором хранятся условия показа (исключительно для улучшения
     *                                           результатов запроса)
     * @param operatorUid                        uid оператора
     * @param clientId                           id клиента
     * @param filter                             настройки фильтрации выбранных условий показа
     * @param orderBy                            настройки упорядочивания условий показа
     * @param statStartDay                       начало периода, за который мы получаем статистику по условиям показа
     * @param statEndDay                         конец периода, за который мы получаем статистику по условиям показа
     * @param goalIds                            идентификаторы целей
     * @param showConditionFetchedFieldsResolver структура содержащая частичную информацию о том, какие поля
     *                                           запрошены на верхнем уровне
     * @param disableStatusFilter                не фильтровать по статусам
     * @param addWithTotalsToQuery               добавляет в запросе with totals с получением общей сатистики
     */
    public GdiShowConditionWithTotals getShowConditions(int shard, Long operatorUid, ClientId clientId,
                                                        GdiShowConditionFilter filter,
                                                        List<GdiShowConditionOrderBy> orderBy,
                                                        LocalDate statStartDay, LocalDate statEndDay,
                                                        Set<Long> goalIds,
                                                        ShowConditionFetchedFieldsResolver showConditionFetchedFieldsResolver,
                                                        boolean disableStatusFilter, boolean addWithTotalsToQuery) {
        //Если для групп достигли лимита, то фразы в базе не фильтруем по adGroupIdIn. Подробнее тут: DIRECT-81384
        if (filter.getAdGroupIdIn() != null && filter.getAdGroupIdIn().size() >= GridAdGroupConstants.getMaxGroupRows()) {
            checkState(isNotEmpty(filter.getCampaignIdIn()),
                    "campaignIdIn from filter must be not empty, when count of adGroupIdIn over limit");
            filter.setAdGroupIdIn(null);
        }

        boolean getRevenueOnlyByAvailableGoals =
                featureService.isEnabledForClientId(clientId, FeatureName.GET_REVENUE_ONLY_BY_AVAILABLE_GOALS);
        Set<Long> availableGoalIds = null;
        if (CollectionUtils.isNotEmpty(goalIds) && getRevenueOnlyByAvailableGoals) {
            availableGoalIds = metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                    operatorUid, clientId);
        }
        return gridShowConditionYtRepository
                .getShowConditions(shard, filter, orderBy, statStartDay, statEndDay,
                        limited(GridShowConditionConstants.getMaxConditionRows()),
                        goalIds, availableGoalIds,
                        showConditionFetchedFieldsResolver, disableStatusFilter, addWithTotalsToQuery);
    }
}
