package ru.yandex.direct.grid.core.entity.smartfilter;

import java.time.LocalDate;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.StreamEx;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.Record;
import org.jooq.Row3;
import org.jooq.SelectHavingStep;
import org.jooq.types.ULong;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.core.entity.smartfilter.model.GdiSmartFilter;
import ru.yandex.direct.grid.core.entity.smartfilter.model.GdiSmartFilterStats;
import ru.yandex.direct.grid.core.util.stats.GridStatNew;
import ru.yandex.direct.grid.core.util.stats.completestat.DirectPhraseStatData;
import ru.yandex.direct.grid.core.util.yt.YtDynamicSupport;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;

import static java.util.Arrays.asList;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.grid.schema.yt.Tables.DIRECTPHRASESTATV2_BS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;

@Repository
@ParametersAreNonnullByDefault
public class GridSmartFilterYtRepository {
    private final GridStatNew<Directphrasestatv2Bs, DirectphrasegoalsstatBs> gridStat
            = new GridStatNew<>(DirectPhraseStatData.INSTANCE);
    private static final Directphrasestatv2Bs STAT = DIRECTPHRASESTATV2_BS.as("S");
    private static final Field<Long> CAMPAIGN_ID = aliased(STAT.EXPORT_ID);
    private static final Field<Long> AD_GROUP_ID = aliased(STAT.GROUP_EXPORT_ID);
    private static final Field<ULong> PHRASE_ID = aliased(STAT.PHRASE_EXPORT_ID);
    private final YtDynamicSupport ytSupport;

    public GridSmartFilterYtRepository(YtDynamicSupport ytSupport) {
        this.ytSupport = ytSupport;
    }

    public Map<Long, GdiSmartFilterStats> getStatistic(List<GdiSmartFilter> smartFilters,
                                                       LocalDate statDateFrom, LocalDate statDateTo,
                                                       Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        Collection<Row3<Long, Long, ULong>> idFilter = smartFilters.stream()
                .map(r -> row(r.getCampaignId(), r.getAdGroupId(), ULong.valueOf(r.getSmartFilterId())))
                .collect(toList());

        Condition idFilterCondition = row(CAMPAIGN_ID, AD_GROUP_ID, PHRASE_ID).in(idFilter);

        SelectHavingStep<Record> selectStep = gridStat.constructStatSelectWithPhraseGoals(
                asList(CAMPAIGN_ID, AD_GROUP_ID, PHRASE_ID),
                idFilterCondition,
                statDateFrom, statDateTo, goalIds, goalIdsForRevenue
        );

        return StreamEx.of(ytSupport.selectRows(selectStep).getYTreeRows())
                .mapToEntry(r -> r.getOrThrow(PHRASE_ID.getName()).longValue(),
                        n -> new GdiSmartFilterStats()
                                .withStat(GridStatNew.addZeros(gridStat.extractStatsEntry(n)))
                                .withGoalStats(gridStat.extractGoalStatEntries(n, goalIds)))
                .toMap();
    }
}
