package ru.yandex.direct.grid.core.entity.statistics.repository;

import java.math.BigDecimal;
import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.SelectJoinStep;
import org.jooq.SortField;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsRequest;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsRow;
import ru.yandex.direct.grid.core.entity.statistics.brandsafety.model.GdiBrandSafetyStatsTotalRequest;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.schema.yt.tables.BrandsafetycategorystatBs;
import ru.yandex.direct.grid.schema.yt.tables.BrandsafetystatBs;
import ru.yandex.direct.ytcomponents.service.BrandSafetyStatsDynContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;

import static java.math.RoundingMode.HALF_UP;
import static java.util.stream.Collectors.toList;
import static org.jooq.impl.DSL.row;
import static org.jooq.impl.DSL.sum;
import static org.jooq.impl.DSL.val;
import static ru.yandex.direct.grid.schema.yt.tables.BrandsafetycategorystatBs.BRANDSAFETYCATEGORYSTAT_BS;
import static ru.yandex.direct.grid.schema.yt.tables.BrandsafetystatBs.BRANDSAFETYSTAT_BS;
import static ru.yandex.direct.ytwrapper.YtTableUtils.aliased;


@Repository
@ParametersAreNonnullByDefault
public class BrandSafetyStatsYtRepository {
    private static final long THREE_HOUR = 10800L;
    private static final long TORRENT_ID = 1024L;
    private static final long ILLEGAL_ID = 1L;

    private static final BrandsafetystatBs BS_STAT = BRANDSAFETYSTAT_BS.as("S");
    private static final BrandsafetycategorystatBs BS_CATEGORY_STAT = BRANDSAFETYCATEGORYSTAT_BS.as("CS");

    private static final Field<Long> DATE = aliased(BS_STAT.DATE);
    private static final Field<Long> ORDER_ID = aliased(BS_STAT.ORDER_ID);
    private static final Field<Long> BANNED = aliased(BS_STAT.BANNED);
    private static final Field<Long> BANNED_UNDEFINED = aliased(BS_STAT.BANNED_UNDEFINED);
    private static final Field<Long> ACCEPTED = aliased(BS_STAT.ACCEPTED);

    private static final Field<Long> CATEGORY = aliased(BS_CATEGORY_STAT.CATEGORY);
    private static final Field<Long> CATEGORY_BANNED = BS_CATEGORY_STAT.BANNED.as(
            "CS" + BS_CATEGORY_STAT.BANNED.getName());

    private static final String PERCENTAGE_NAME = "Percentage";
    private static final String PERIOD_NAME = "Period";

    // BANNED_UNDEFINED пока что игнорируется: https://st.yandex-team.ru/BSSERVER-17045#60af4c633a6fa04f5f74a2fe
    private static final Field<BigDecimal> PERCENTAGE =
            YtDSL.toDouble(BANNED)
                    .mul(100)
                    .div(YtDSL.toDouble(BANNED.add(ACCEPTED)))
                    .as(PERCENTAGE_NAME);

    private static final Field<BigDecimal> CATEGORY_PERCENTAGE =
            YtDSL.toDouble(CATEGORY_BANNED)
                    .mul(100)
                    .div(YtDSL.toDouble(BANNED.add(ACCEPTED)))
                    .as(PERCENTAGE_NAME);

    private static final Field<Long> ONE = val(1L);

    private final BrandSafetyStatsDynContextProvider dynContextProvider;

    public BrandSafetyStatsYtRepository(BrandSafetyStatsDynContextProvider dynContextProvider) {
        this.dynContextProvider = dynContextProvider;
    }

    public List<GdiBrandSafetyStatsRow> getStats(GdiBrandSafetyStatsRequest statsRequest) {
        var select = YtDSL.ytContext().select(
                getPeriodField(statsRequest).as(PERIOD_NAME),
                sum(BS_STAT.BANNED).as(BANNED),
                sum(BS_STAT.BANNED_UNDEFINED).as(BANNED_UNDEFINED),
                sum(BS_STAT.ACCEPTED).as(ACCEPTED));

        if (statsRequest.getShowCampaigns()) {
            select.select(BS_STAT.ORDER_ID.as(ORDER_ID)).groupBy(ORDER_ID);
        }

        final SelectJoinStep<?> queryJoinStep;
        if (statsRequest.getShowCategories()) {
            select.select(sum(BS_CATEGORY_STAT.BANNED).as(CATEGORY_BANNED));
            select.select(BS_CATEGORY_STAT.CATEGORY.as(CATEGORY)).groupBy(CATEGORY);

            // порядок джойна важен, так как если джойнить BS_CATEGORY_STAT к BS_STAT,
            // то YT отказывается использовать индексы
            queryJoinStep = select
                    .select(CATEGORY_PERCENTAGE)
                    .from(BS_CATEGORY_STAT)
                    .join(BS_STAT)
                    .on(row(BS_CATEGORY_STAT.DATE, BS_CATEGORY_STAT.ORDER_ID).eq(row(BS_STAT.DATE, BS_STAT.ORDER_ID)));
        } else {
            queryJoinStep = select
                    .select(PERCENTAGE)
                    .from(BS_STAT);
        }

        var query = queryJoinStep
                .where(BS_STAT.DATE.between(statsRequest.getStartDate(), statsRequest.getEndDate()))
                .and(BS_STAT.ORDER_ID.in(statsRequest.getOrderIds()));
        if (statsRequest.getShowCategories()) {
            // В разрезе по категориям не должны отображаться незаконный контент и торренты.
            // Юридическая особенность. Комплексно решим в DIRECT-147877
            query.andNot(BS_CATEGORY_STAT.CATEGORY.in(ILLEGAL_ID, TORRENT_ID));
        }

        if (statsRequest.getOrderBy() != null) {
            query.orderBy(getSortField(statsRequest));
        }

        if (statsRequest.getLimitOffset() != null) {
            query.offset(statsRequest.getLimitOffset().getOffset())
                    .limit(statsRequest.getLimitOffset().getLimit());
        }

        //группировка по периодам
        query.groupBy(DATE.as(PERIOD_NAME));

        var rows = dynContextProvider.getContext().executeSelect(query).getYTreeRows();
        return rows.stream()
                .map(this::readBrandSafetyStatsRow)
                .collect(toList());
    }

    private Field<Long> getPeriodField(GdiBrandSafetyStatsRequest statsRequest) {
        switch (statsRequest.getGroupBy()) {
            case WEEK:
                return YtDSL.unixEpochStartOfWeek(BS_STAT.DATE, THREE_HOUR);

            case MONTH:
                return YtDSL.unixEpochStartOfMonth(BS_STAT.DATE, THREE_HOUR);

            case QUARTER:
                return YtDSL.unixEpochStartOfQuarter(BS_STAT.DATE, THREE_HOUR);

            case YEAR:
                return YtDSL.unixEpochStartOfYear(BS_STAT.DATE, THREE_HOUR);

            case SELECTED_PERIOD:
                return ONE;

            default:
                return BS_STAT.DATE;
        }
    }

    private SortField<?> getSortField(GdiBrandSafetyStatsRequest statsRequest) {
        final Field<?> orderBy;
        switch (statsRequest.getOrderBy().getField()) {
            case CID:
                orderBy = ORDER_ID;
                break;

            case PERCENTAGE:
                orderBy = statsRequest.getShowCategories() ? CATEGORY_PERCENTAGE : PERCENTAGE;
                break;

            default:
                orderBy = DATE.as(PERIOD_NAME);
                break;
        }

        return Order.DESC.equals(statsRequest.getOrderBy().getOrder()) ? orderBy.desc() : orderBy.asc();
    }

    public Double getStatsTotal(GdiBrandSafetyStatsTotalRequest statsTotalRequest) {
        var query = YtDSL.ytContext()
                .select(
                        sum(BS_STAT.BANNED).as(BANNED),
                        sum(BS_STAT.BANNED_UNDEFINED).as(BANNED_UNDEFINED),
                        sum(BS_STAT.ACCEPTED).as(ACCEPTED),
                        PERCENTAGE)
                .from(BS_STAT)
                .where(BS_STAT.DATE.between(statsTotalRequest.getStartDate(), statsTotalRequest.getEndDate()))
                .and(BS_STAT.ORDER_ID.in(statsTotalRequest.getOrderIds()))
                // https://yt.yandex-team.ru/docs/description/dynamic_tables/dyn_query_language.html#primery
                .groupBy(ONE);

        var rows = dynContextProvider.getContext().executeSelect(query).getYTreeRows();
        if (rows.isEmpty()) {
            return 0.0;
        }

        return getPercentageValue(rows.get(0));
    }

    private GdiBrandSafetyStatsRow readBrandSafetyStatsRow(YTreeMapNode node) {
        return new GdiBrandSafetyStatsRow()
                .withDate(node.getLong(PERIOD_NAME))
                .withOrderId(node.getLongO(ORDER_ID.getName()).orElse(null))
                .withCategory(node.getLongO(CATEGORY.getName()).orElse(null))
                .withBannedCount(node.getLongO(CATEGORY_BANNED.getName())
                        .orElse(node.getLongO(BANNED.getName()).orElse(null)))
                .withBannedPercentage(getPercentageValue(node));
    }

    private static Double getPercentageValue(YTreeMapNode node) {
        return BigDecimal.valueOf(node.getDoubleO(PERCENTAGE_NAME).orElse(0.0))
                .setScale(2, HALF_UP)
                .doubleValue();
    }
}
