package ru.yandex.direct.grid.core.entity.strategy

import org.springframework.stereotype.Service
import ru.yandex.direct.core.entity.campaign.model.CampaignWithPackageStrategy
import ru.yandex.direct.core.entity.campaign.repository.CampaignTypedRepository
import ru.yandex.direct.core.entity.container.LocalDateRange
import ru.yandex.direct.core.entity.metrika.service.MetrikaGoalsService
import ru.yandex.direct.dbschema.ppc.tables.Campaigns
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.dbutil.sharding.ShardHelper
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService.getAggregatorGoalIdBySubGoalId
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignService.sumGoalStatForAggregatedGoals
import ru.yandex.direct.grid.core.entity.campaign.service.GridCampaignServiceUtils
import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats
import ru.yandex.direct.grid.core.entity.model.GdiGoalConversion
import ru.yandex.direct.grid.core.entity.model.campaign.GdiAggregatorGoal
import ru.yandex.direct.grid.core.entity.model.campaign.GdiCampaignStats
import ru.yandex.direct.grid.core.entity.strategy.repository.GridPackageStrategyYtRepository
import ru.yandex.direct.grid.core.util.stats.GridStatUtils
import ru.yandex.direct.multitype.repository.filter.ConditionFilterFactory.whereEqFilter
import ru.yandex.direct.utils.DateTimeUtils
import java.time.LocalDate

@Service
class GridStatPackageStrategyService(
    private val gridStatRepository: GridPackageStrategyYtRepository,
    private val campaignTypedRepository: CampaignTypedRepository,
    private val shardHelper: ShardHelper,
    private val metrikaGoalsService: MetrikaGoalsService
) {

    /**
     * Метод получения статистики по целям стратегии.
     * Статистика фильтруется по целям привязанным к стратегии
     * */
    fun getStatsForAssignedGoals(
        clientId: ClientId,
        operatorUid: Long,
        strategyIds: Set<Long>,
        startDate: LocalDate,
        endDate: LocalDate,
        revenueOnlyByAvailableGoals: Boolean
    ): Map<Long, GdiCampaignStats> {
        val availableGoals = availableGoals(clientId, operatorUid, revenueOnlyByAvailableGoals)
        val allCampaignIds = allClientCampaigns(clientId)

        val filteringTuples = gridStatRepository.getFilteringTuples(
            allCampaignIds.map { it to LocalDateRange().withFromInclusive(startDate).withToInclusive(endDate) }.toMap(),
            strategyIds
        )
        val filter = GridPackageStrategyYtRepository.Filter(filteringTuples)
        val stats = gridStatRepository.strategyEntityStats(filter)
        val goalConversions = gridStatRepository.strategyGoalsConversions(filter, true, availableGoals)

        val entityStatConversions = goalConversions.mapValues {
            val zero = GdiEntityConversion()
                .withGoals(0)
                .withRevenue(0)
            it.value.fold(zero) { entityConversion, goalConversion ->
                entityConversion.revenue += goalConversion.revenue
                entityConversion.goals += goalConversion.goals
                entityConversion
            }
        }

        stats.forEach { (id, stat) ->
            GridStatUtils.updateStatWithConversions(
                stat,
                entityStatConversions[id]
            )
        }
        return stats.mapValues {
            GdiCampaignStats()
                .withStat(it.value)
        }
    }

    /**
     * Метод получения статистики по целям стратегий.
     * В качестве фильтрации используеются цели переданные в метод.
     * */
    fun getStatsByGoals(
        clientId: ClientId,
        operatorUid: Long,
        strategyIds: Set<Long>,
        startDate: LocalDate,
        endDate: LocalDate,
        goalIds: Set<Long>,
        revenueOnlyByAvailableGoals: Boolean
    ): Map<Long, GdiCampaignStats> {
        val availableGoals = availableGoals(clientId, operatorUid, revenueOnlyByAvailableGoals)
        val allCampaignIds = allClientCampaigns(clientId)

        val filteringTuples = gridStatRepository.getFilteringTuples(
            allCampaignIds.map { it to LocalDateRange().withFromInclusive(startDate).withToInclusive(endDate) }.toMap(),
            strategyIds
        )
        val filter =
            GridPackageStrategyYtRepository.Filter(filteringTuples, strategyIds.map { it to goalIds.toList() }.toMap())

        val entityStats = gridStatRepository.strategyEntityStats(filter)

        val goalConversions = gridStatRepository.strategyGoalsConversions(filter, false, availableGoals)

        val goalStats = GridCampaignServiceUtils.calculateCampaignsGoalStat(
            goalConversions,
            entityStats
        )
        return entityStats.mapValues {
            val result = GdiCampaignStats()
                .withStat(it.value)

            val goalStat = goalStats[it.key]
            if (goalStat != null) {
                result.withGoalStats(goalStat)
            }
            result
        }
    }

    private fun availableGoals(
        clientId: ClientId,
        operatorUid: Long,
        revenueOnlyByAvailableGoals: Boolean
    ): Set<Long>? {
        return if (revenueOnlyByAvailableGoals) {
            metrikaGoalsService.getAvailableMetrikaGoalIdsForClientWithExceptionHandling(
                operatorUid, clientId
            )?.toSet()
        } else {
            null
        }
    }

    /**
     * Метод для получения статистики по стратегиям
     * В качестве фильтраций используются переданные промежутки времени для стратегий и переданные цели
     */
    fun getStatsByStrategyIdsWithDifferentDateRange(
        clientId: ClientId,
        dateRangesByStrategyId: Map<Long, LocalDateRange>,
        aggregatorGoalByStrategyId: Map<Long, GdiAggregatorGoal>,
        strategyGoalIdsByStrategyId: Map<Long, Long>
    ): Map<Long, GdiCampaignStats> {
        val allCampaignIds = allClientCampaigns(clientId)
        val strategyIds = dateRangesByStrategyId.keys
        val from = dateRangesByStrategyId.values.minOf(LocalDateRange::getFromInclusive)
        val to = LocalDate.now()
        val filteringTuples = gridStatRepository.getFilteringTuples(
            allCampaignIds.map { it to LocalDateRange().withFromInclusive(from).withToInclusive(to) }.toMap(),
            strategyIds
        )
        val filteredByUpdateTime = filteringTuples.filter { tuple ->
            dateRangesByStrategyId[tuple.effecitveStrategyId]?.contains(tuple.updateTime) ?: false
        }

        val availableGoalsByCid = aggregatorGoalByStrategyId.mapValues { (_, aggregatorGoal) ->
            aggregatorGoal.subGoalIds ?: listOf()
        }

        val filter = GridPackageStrategyYtRepository.Filter(
            filteredByUpdateTime,
            availableGoalsByCid
        )

        val strategyEntityStatsByStrategyId = gridStatRepository.strategyEntityStats(filter)

        val goalConversionsByStrategyId = gridStatRepository.strategyGoalsConversions(filter, false)
        val mergedGoalConversionsByStrategyId = goalConversionsByStrategyId.mapValues { (strategyId, stats) ->
            sumGoalStatForAggregatedGoals(
                getAggregatorGoalIdBySubGoalId(listOf(aggregatorGoalByStrategyId[strategyId])),
                stats
            )
        }

        return getStrategyStatsByStrategyId(
            mergedGoalConversionsByStrategyId,
            strategyEntityStatsByStrategyId,
            strategyGoalIdsByStrategyId
        )
    }

    private fun LocalDateRange.contains(updateTime: Long): Boolean =
        this.fromInclusive.atStartOfDay(DateTimeUtils.MSK).toEpochSecond() <= updateTime &&
            this.toInclusive.atStartOfDay(DateTimeUtils.MSK).toEpochSecond() >= updateTime

    private fun allClientCampaigns(clientId: ClientId): Set<Long> {
        val shard = shardHelper.getShardByClientId(clientId)
        val campaignIds = campaignTypedRepository.getSafely(
            shard,
            whereEqFilter(Campaigns.CAMPAIGNS.CLIENT_ID, clientId.asLong()),
            CampaignWithPackageStrategy::class.java
        ).map { it.id }
        return campaignIds.toSet()
    }

    private fun getStrategyStatsByStrategyId(
        goalConversionsByStrategyId: Map<Long, List<GdiGoalConversion>>,
        strategyEntityStatsByStrategyId: Map<Long, GdiEntityStats>,
        strategyGoalIdsByStrategyId: Map<Long, Long>
    ): Map<Long, GdiCampaignStats> {
        val strategyGoalsStatByStrategyId =
            GridCampaignServiceUtils.calculateCampaignsGoalStat(
                goalConversionsByStrategyId,
                strategyEntityStatsByStrategyId
            )

        return strategyGoalIdsByStrategyId.mapValues { (strategyId, goalId) ->
            val entityStats = strategyEntityStatsByStrategyId[strategyId]
            GridCampaignService.getCampaignStatsOrDefaultZeroStats(
                strategyGoalsStatByStrategyId,
                strategyId,
                entityStats,
                setOf(goalId)
            )
        }
    }
}
