package ru.yandex.direct.grid.core.entity.sync.service;


import java.time.Duration;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

import javax.annotation.ParametersAreNonnullByDefault;

import com.github.shyiko.mysql.binlog.GtidSet;
import one.util.streamex.StreamEx;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.grid.core.entity.sync.model.GdiClientMutationState;
import ru.yandex.direct.grid.core.entity.sync.repository.YtSyncStateRepository;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.monlib.metrics.histogram.Histograms;
import ru.yandex.monlib.metrics.primitives.Histogram;
import ru.yandex.monlib.metrics.primitives.Rate;

import static com.google.common.base.Preconditions.checkNotNull;
import static com.google.common.collect.Iterables.getFirst;
import static ru.yandex.direct.solomon.SolomonUtils.SOLOMON_REGISTRY;
import static ru.yandex.direct.utils.ThreadUtils.sleep;

/**
 * Сервис для получения данных о состоянии синхранизации mysql баз в YT'е
 */
@Service
@ParametersAreNonnullByDefault
public class YtSyncStateService {

    private static final Logger logger = LoggerFactory.getLogger(YtSyncStateService.class);

    private static final Histogram SYNC_TIME_SENSOR = SOLOMON_REGISTRY.histogramRate(
            "yt_sync_spent_time_millis",
            Histograms.exponential(12, 2, 100)
    );

    private static final Map<GdiClientMutationState, Rate> SENSOR_BY_STATE = initStateSensors();

    private final int maxAttemptsToCheckGtidSet;
    private final Duration sleepDurationBetweenTries;
    private final ShardHelper shardHelper;
    private final YtSyncStateRepository ytSyncStateRepository;

    @Autowired
    public YtSyncStateService(DirectYtDynamicConfig dynamicConfig,
                              ShardHelper shardHelper,
                              YtSyncStateRepository ytSyncStateRepository) {
        this.shardHelper = shardHelper;
        this.ytSyncStateRepository = ytSyncStateRepository;

        DirectYtDynamicConfig.CheckSyncStateConfig checkSyncStateConfig = dynamicConfig.checkSyncStateConfig();
        this.maxAttemptsToCheckGtidSet = checkSyncStateConfig.maxAttemptsToCheckGtidSet();
        this.sleepDurationBetweenTries = checkSyncStateConfig.sleepDurationBetweenTries();
    }


    /**
     * Возвращает состояние синронизации изменений в YT'е соответствующий переданному gtid_set
     * <p>
     * Возвращает {@link GdiClientMutationState#UNKNOWN}, если не удалось получить состояние синхранизации в YT'е
     */
    public GdiClientMutationState checkClientMutationStateInYt(String login, String mysqlCurrentGtidSet) {
        int shard = shardHelper.getShardByLoginStrictly(login);

        GtidSet mysqlGtidSet = new GtidSet(mysqlCurrentGtidSet);
        GtidSet.UUIDSet mysqlUUIDSet = getFirst(mysqlGtidSet.getUUIDSets(), null);
        checkNotNull(mysqlUUIDSet, "mysqlUUIDSet should not be null");

        long startTime = System.nanoTime();
        var result = waitUntilYtGtidSetContainedWithin(shard, mysqlUUIDSet);
        long spentTimeMillis = TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - startTime);

        if (result != GdiClientMutationState.SYNCED) {
            logger.warn("got {} sync state", result);
        }
        recordMetrics(result, spentTimeMillis);
        return result;
    }

    private GdiClientMutationState waitUntilYtGtidSetContainedWithin(int clientShard, GtidSet.UUIDSet mysqlUuidSet) {
        GdiClientMutationState result = GdiClientMutationState.UNKNOWN;

        for (int attemptNumber = 0; attemptNumber < maxAttemptsToCheckGtidSet; attemptNumber++) {
            try {
                Optional<String> gtidSet = ytSyncStateRepository.getGtidSet(clientShard);
                GtidSet.UUIDSet uuidSet = gtidSet.map(GtidSet::new)
                        .map(gs -> gs.getUUIDSet(mysqlUuidSet.getUUID()))
                        .orElse(null);

                if (mysqlUuidSet.isContainedWithin(uuidSet)) {
                    return GdiClientMutationState.SYNCED;
                }

                if (uuidSet != null) {
                    result = GdiClientMutationState.NOT_SYNCED;
                }
            } catch (Exception e) {
                logger.error("Got exception while fetching gtid_set", e);
            }

            sleep(sleepDurationBetweenTries);
        }

        return result;
    }

    private void recordMetrics(GdiClientMutationState state, long spentTimeMillis) {
        SYNC_TIME_SENSOR.record(spentTimeMillis);
        SENSOR_BY_STATE.get(state).inc();
    }

    private static Map<GdiClientMutationState, Rate> initStateSensors() {
        return StreamEx.of(GdiClientMutationState.values())
                .mapToEntry(state -> SOLOMON_REGISTRY.rate("yt_sync_state_" + state.name().toLowerCase()))
                .toMap();
    }

}
