package ru.yandex.direct.grid.core.entity.touchsocdem.service.converter;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.bidmodifier.AgeType;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographics;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierDemographicsAdjustment;
import ru.yandex.direct.core.entity.bidmodifier.BidModifierType;
import ru.yandex.direct.core.entity.bidmodifier.GenderType;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdiTouchSocdem;
import ru.yandex.direct.grid.processing.model.touchsocdem.GdiTouchSocdemAgePoint;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class GridTouchSocdemConverter {
    private GridTouchSocdemConverter() {
    }

    /**
     * Соцдем таргетинг на всех.
     */
    public static GdiTouchSocdem emptySocdem() {
        return new GdiTouchSocdem()
                .withGenders(Collections.emptyList())
                .withAgeLower(GdiTouchSocdemAgePoint.AGE_0)
                .withAgeUpper(GdiTouchSocdemAgePoint.AGE_INF)
                .withIncompleteBidModifier(false);
    }

    /**
     * Перевод соцдем корректировки в описание тачёвого соцдем таргетинга.
     * <p>
     * Если корректировку не удалось перевести в тачёвый соцдем без потери информации
     * (например есть корректировки отличные от -100%), результат будет иметь больший охват,
     * и в объекте-описании будет выставлен флаг {@code incompleteBidModifier}
     */
    public static GdiTouchSocdem socdemBidModifierToTouchSocdem(@Nullable BidModifierDemographics bidModifier) {
        if (bidModifier == null) {
            return emptySocdem();
        }

        GdiTouchSocdem result = new GdiTouchSocdem();

        Map<GenderType, EnumSet<AgeType>> targets = getFullTargetsMap();
        excludeZeroModifiersFromTargets(targets, bidModifier.getDemographicsAdjustments());

        result.setGenders(extractGendersFromTargetMap(targets));
        var agePointPair = extractAgePointsFromTargetMap(targets);
        result.setAgeLower(agePointPair.lowerPoint);
        result.setAgeUpper(agePointPair.upperPoint);
        result.setIncompleteBidModifier(isTouchSocdemBackwardsIncompatible(result, bidModifier));

        return result;
    }

    /**
     * Карта таргетов, где включен таргетинг на все возраста всех гендеров.
     * Неизвестный возраст не включен в карту для удобства, т.к. на него всегда есть неявный таргетинг.
     * (за исключением случаев, когда выключен таргетинг на какой-то гендер целиком)
     */
    private static Map<GenderType, EnumSet<AgeType>> getFullTargetsMap() {
        EnumSet<AgeType> allAges = EnumSet.allOf(AgeType.class);
        // период 45+ больше не используется, вместо него 45-55 + 55+
        allAges.remove(AgeType._45_);
        // на неизвестный возраст нельзя выставить корректировку через веб интерфейс, он всегда неявно включен
        // в таргетинг (за исключением случаев, когда выключен таргетинг на какой-то гендер целиком)
        allAges.remove(AgeType.UNKNOWN);
        Map<GenderType, EnumSet<AgeType>> targets = new HashMap<>();
        for (GenderType g : GenderType.values()) {
            targets.put(g, EnumSet.copyOf(allAges));
        }
        return targets;
    }

    /**
     * "выкусываем" из карты таргетов те, которые исключают корректировки с 0%
     */
    private static void excludeZeroModifiersFromTargets(
            Map<GenderType, EnumSet<AgeType>> targets,
            List<BidModifierDemographicsAdjustment> demographicsAdjustments
    ) {
        demographicsAdjustments.stream()
                .filter(adj -> adj.getPercent() == 0)
                .forEach(adj -> {
                    List<GenderType> genders = new ArrayList<>();
                    if (adj.getGender() == null) {
                        genders.addAll(List.of(GenderType.values()));
                    } else {
                        genders.add(adj.getGender());
                    }

                    for (GenderType gender : genders) {
                        if (adj.getAge() == null) {
                            targets.get(gender).clear();
                        } else {
                            targets.get(gender).remove(adj.getAge());
                        }
                    }
                });

        for (GenderType g : GenderType.values()) {
            if (targets.get(g).isEmpty()) {
                targets.remove(g);
            }
        }
    }

    /**
     * Вычисляет из карты таргетов гендер, на который включено таргетирование.
     * Если таргеты есть у нескольких гендеров, или ни у однго, считаем что таргетирования по гендеру нет,
     * (таргетируемся на всех) и возвращаем пустой список.
     */
    private static List<GenderType> extractGendersFromTargetMap(Map<GenderType, EnumSet<AgeType>> targets) {
        List<GenderType> gendersWithTargets = targets.keySet().stream()
                .filter(gender -> !targets.get(gender).isEmpty())
                .collect(Collectors.toList());
        if (gendersWithTargets.size() != 1) {
            return Collections.emptyList();
        } else {
            return gendersWithTargets;
        }
    }

    /**
     * Вычисляет из карты таргетов нижнюю и верхнюю границы соцдем таргетинга.
     * Если корректировка выключает таргетинг на все возраста, считаем, что таргет включен на всех.
     */
    private static AgeRange extractAgePointsFromTargetMap(Map<GenderType, EnumSet<AgeType>> targets) {
        EnumSet<AgeType> allTargetAges = EnumSet.noneOf(AgeType.class);
        for (EnumSet<AgeType> ages : targets.values()) {
            allTargetAges.addAll(ages);
        }
        if (allTargetAges.isEmpty()) {
            return new AgeRange(GdiTouchSocdemAgePoint.AGE_0, GdiTouchSocdemAgePoint.AGE_INF);
        }
        ArrayList<AgeRange> allAgePoints = new ArrayList<>(mapList(allTargetAges, AgeRange::fromAgeType));
        allAgePoints.sort(Comparator.comparingInt(app -> app.lowerPoint.getTypedValue()));
        return new AgeRange(
                allAgePoints.get(0).lowerPoint,
                allAgePoints.get(allAgePoints.size() - 1).upperPoint
        );
    }

    /**
     * @return {@code false}, если тачёвый соцдем {@code touchSocdem} можно перевести в соцдем корректировку,
     * и она будет такой же, как корректировка {@code bidModifiers}
     */
    private static boolean isTouchSocdemBackwardsIncompatible(
            GdiTouchSocdem touchSocdem, @Nullable BidModifierDemographics bidModifier
    ) {
        BidModifierDemographics fromSocdem = toSocdemBidModifier(touchSocdem);
        if (fromSocdem == null && bidModifier == null) {
            return false;
        }
        if (fromSocdem == null || bidModifier == null) {
            return true;
        }
        if (fromSocdem.getType() != bidModifier.getType() || fromSocdem.getEnabled() != bidModifier.getEnabled()) {
            return true;
        }

        Set<BidModifierDemographicsAdjustment> adjFromSocdem = new HashSet<>(fromSocdem.getDemographicsAdjustments());
        Set<BidModifierDemographicsAdjustment> adjFromBidModifier = new HashSet<>(
                mapList(bidModifier.getDemographicsAdjustments(), GridTouchSocdemConverter::extractImportantAdjFields)
        );

        return !adjFromSocdem.equals(adjFromBidModifier);
    }

    /**
     * Создаёт новый объект {@code BidModifierDemographicsAdjustment} с набором полей из {@code adj}, которые влияют
     * на корректировку (т.к. без айдишников, {@code lastChange} и пр.)
     */
    private static BidModifierDemographicsAdjustment extractImportantAdjFields(BidModifierDemographicsAdjustment adj) {
        return new BidModifierDemographicsAdjustment()
                .withAge(adj.getAge())
                .withGender(adj.getGender())
                .withPercent(adj.getPercent());
    }

    /**
     * Создаёт соцдем корректировку из тачёвого соцдем таргетинга.
     * Это достигается созданием корректировок на -100% всем сочетаниям гендеров и возрастов, которые не входят в
     * тачёвый соцдем.
     * <p>
     * Неизвестный возраст не исключается, т.к. в веб интерфейсе не поддержана такая настройка.
     * <p>
     * Если соцдем таргетинг отсутствует (таргет на всех), возвращает {@code null}.
     * <p>
     * NB! Пустой список гендеров в таргетинге означает таргетинг на все гендеры.
     */
    public static @Nullable
    BidModifierDemographics toSocdemBidModifier(@Nullable GdiTouchSocdem touchSocdem) {
        if (touchSocdem == null) {
            return null;
        }
        var bmd = new BidModifierDemographics()
                .withType(BidModifierType.DEMOGRAPHY_MULTIPLIER)
                .withEnabled(true);
        EnumSet<GenderType> genders = EnumSet.noneOf(GenderType.class);
        genders.addAll(touchSocdem.getGenders());
        EnumSet<GenderType> invGenders = EnumSet.complementOf(genders);
        boolean emptyGenders = false;
        if (genders.isEmpty() || invGenders.isEmpty()) {
            emptyGenders = true;
            genders = null;
            invGenders = null;
        }

        EnumSet<AgeType> ages = EnumSet.noneOf(AgeType.class);
        ages.addAll(convertAges(touchSocdem.getAgeLower(), touchSocdem.getAgeUpper()));
        EnumSet<AgeType> invAges = EnumSet.complementOf(ages);
        // период 45+ больше не используется, вместо него 45-55 + 55+
        invAges.remove(AgeType._45_);
        // неизвестный возраст не поддерживается в веб интерфейсе, поэтому не исключаем его корректировкой
        invAges.remove(AgeType.UNKNOWN);
        boolean emptyAges = false;
        if (ages.isEmpty() || invAges.isEmpty()) {
            emptyAges = true;
            ages = null;
            invAges = null;
        }

        if (emptyGenders && emptyAges) {
            return null;
        }

        ArrayList<BidModifierDemographicsAdjustment> adjList = new ArrayList<>();
        if (emptyGenders) {
            // таргетируемся на все гендеры определённых возрастов
            for (AgeType age : invAges) {
                adjList.add(new BidModifierDemographicsAdjustment()
                        .withGender(null)
                        .withAge(age)
                        .withPercent(0));
            }
        } else {
            for (GenderType gender : invGenders) {
                // на этот гендер не таргетируемся в любом возрасте
                adjList.add(new BidModifierDemographicsAdjustment()
                        .withGender(gender)
                        .withAge(null)
                        .withPercent(0));
            }
            if (!emptyAges) {
                for (GenderType gender : genders) {
                    // есть таргетинг одновременно на гендер и на диапазон возрастов
                    // выключаем показы у возрастов не включенных в таргет (гендеры уже выключили выше)
                    for (AgeType age : invAges) {
                        adjList.add(new BidModifierDemographicsAdjustment()
                                .withGender(gender)
                                .withAge(age)
                                .withPercent(0));
                    }
                }
            }
        }

        bmd.setDemographicsAdjustments(adjList);
        return bmd;
    }

    /**
     * Возвращает список значений енума {@code AgeType}, которые соответствуют возрастам, входящим в диапазон
     * {@code [ageLower, ageUpper)}
     */
    private static List<AgeType> convertAges(GdiTouchSocdemAgePoint ageLower, GdiTouchSocdemAgePoint ageUpper) {
        AgeRange socdemAgeRange = new AgeRange(ageLower, ageUpper);
        return Arrays.stream(AgeType.values())
                .filter(age -> age != AgeType._45_ && age != AgeType.UNKNOWN)
                .filter(age -> socdemAgeRange.includes(AgeRange.fromAgeType(age)))
                .collect(Collectors.toList());
    }

    private static class AgeRange {
        final GdiTouchSocdemAgePoint lowerPoint;
        final GdiTouchSocdemAgePoint upperPoint;

        AgeRange(GdiTouchSocdemAgePoint lowerPoint, GdiTouchSocdemAgePoint upperPoint) {
            if (upperPoint.getTypedValue() < lowerPoint.getTypedValue()) {
                this.lowerPoint = upperPoint;
                this.upperPoint = lowerPoint;
            } else {
                this.lowerPoint = lowerPoint;
                this.upperPoint = upperPoint;
            }
        }

        static AgeRange fromAgeType(AgeType ageType) {
            switch (ageType) {
                case _0_17:
                    return new AgeRange(GdiTouchSocdemAgePoint.AGE_0, GdiTouchSocdemAgePoint.AGE_18);

                case _18_24:
                    return new AgeRange(GdiTouchSocdemAgePoint.AGE_18, GdiTouchSocdemAgePoint.AGE_25);

                case _25_34:
                    return new AgeRange(GdiTouchSocdemAgePoint.AGE_25, GdiTouchSocdemAgePoint.AGE_35);

                case _35_44:
                    return new AgeRange(GdiTouchSocdemAgePoint.AGE_35, GdiTouchSocdemAgePoint.AGE_45);

                case _45_54:
                    return new AgeRange(GdiTouchSocdemAgePoint.AGE_45, GdiTouchSocdemAgePoint.AGE_55);

                case _55_:
                    return new AgeRange(GdiTouchSocdemAgePoint.AGE_55, GdiTouchSocdemAgePoint.AGE_INF);

                case _45_:
                case UNKNOWN:
                default:
                    throw new RuntimeException("Can't extract age period points from AgeType " + ageType);
            }
        }

        boolean includes(AgeRange otherRange) {
            return this.lowerPoint.getTypedValue() <= otherRange.lowerPoint.getTypedValue()
                    && this.upperPoint.getTypedValue() >= otherRange.upperPoint.getTypedValue();
        }
    }
}
