package ru.yandex.direct.grid.core.frontdb.jsonsettings.service;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ObjectWriter;
import com.jayway.jsonpath.Configuration;
import com.jayway.jsonpath.JsonPath;
import com.jayway.jsonpath.Option;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Service;

import ru.yandex.direct.grid.core.frontdb.jsonsettings.service.model.FeedbackFormSessionsData;
import ru.yandex.direct.grid.core.frontdb.repository.JsonSettingsRepository;
import ru.yandex.direct.grid.model.jsonsettings.GdGetJsonSettings;
import ru.yandex.direct.grid.model.jsonsettings.GdSetJsonSettings;
import ru.yandex.direct.grid.model.jsonsettings.GdUpdateJsonSettingsUnion;
import ru.yandex.direct.grid.model.jsonsettings.IdType;

import static ru.yandex.direct.utils.FunctionalUtils.filterAndMapList;

@Lazy
@Service
@ParametersAreNonnullByDefault
public class JsonSettingsService {

    private final JsonSettingsRepository jsonSettingsRepository;

    @Autowired
    public JsonSettingsService(JsonSettingsRepository jsonSettingsRepository) {
        this.jsonSettingsRepository = jsonSettingsRepository;
    }

    public String setJsonSettings(@Nullable Long clientId, Long operatorUid, GdSetJsonSettings input) {
        if (input.getIdType() == IdType.UID) {
            clientId = null;
        } else if (input.getIdType() == IdType.CLIENT_ID) {
            operatorUid = null;
        }
        var validItems = filterAndMapList(input.getUpdateItems(),
                item -> isValidJsonPath(item.getJsonPath()),
                item -> Pair.of(item.getJsonPath(), item.getNewValue()));

        String currentSettings = jsonSettingsRepository.getJsonSettings(clientId, operatorUid, List.of("$")).get(0);
        var conf =
                Configuration.defaultConfiguration().addOptions(Option.DEFAULT_PATH_LEAF_TO_NULL).addOptions(Option.SUPPRESS_EXCEPTIONS);
        var json = JsonPath.using(conf).parse(currentSettings);
        for (var item : validItems) {
            /*
            нужно поскольку JsonPath не работает с корневым jsonpath
             */
            if (item.getKey().equals("$")) {
                try {
                    ObjectWriter ow = new ObjectMapper().writer().withDefaultPrettyPrinter();
                    String newJson = ow.writeValueAsString(item.getValue());
                    json = JsonPath.using(conf).parse(newJson);
                } catch (Exception e) {
                    return "";
                }
            } else {
                json = json.set(item.getKey(), item.getValue());
            }
        }
        jsonSettingsRepository.updateJsonSettings(clientId, operatorUid, json.jsonString());
        return json.jsonString();
    }

    public List<String> getJsonSettings(@Nullable Long clientId, Long operatorUid, GdGetJsonSettings input) {
        if (input.getIdType() == IdType.UID) {
            clientId = null;
        } else if (input.getIdType() == IdType.CLIENT_ID) {
            operatorUid = null;
        }
        return jsonSettingsRepository.getJsonSettings(clientId, operatorUid, input.getJsonPath());
    }

    private static boolean isValidJsonPath(String jsonPath) {
        try {
            JsonPath.compile(jsonPath);
        } catch (Exception e) {
            return false;
        }
        return true;
    }

    /**
     * Сбросить даты сессий и запомнить флагом, что надо показать ФОС на третью следующую сессию
     * @param operatorUid — uid оператора запроса
     */
    public void feedbackRateLater(Long operatorUid) {
        var updateItems = addRootFeedbackKey(operatorUid);
        updateItems.addAll(List.of(
                new GdUpdateJsonSettingsUnion()
                        .withNewValue(null)
                        .withJsonPath(FeedbackFormSessionsData.OLDEST_SESSION_DATE),
                new GdUpdateJsonSettingsUnion()
                        .withNewValue(null)
                        .withJsonPath(FeedbackFormSessionsData.SECOND_SESSION_DATE),
                new GdUpdateJsonSettingsUnion()
                        .withNewValue("true")
                        .withJsonPath(FeedbackFormSessionsData.RATE_LATER)
        ));
        var input = new GdSetJsonSettings()
                .withIdType(IdType.UID)
                .withUpdateItems(updateItems);
        setJsonSettings(null, operatorUid, input);
    }

    /**
     * Сбросить счётчики, поставить дату показа ФОС
     * @param operatorUid — uid оператора
     * @param time — новая дата и время показа ФОС (время сейчас)
     */
    public void feedbackMarkAsShown(Long operatorUid, LocalDateTime time) {
        var updateItems = addRootFeedbackKey(operatorUid);
        var timeLong = time.toInstant(ZoneOffset.UTC).getEpochSecond();
        updateItems.addAll(List.of(
                new GdUpdateJsonSettingsUnion()
                        .withNewValue(null)
                        .withJsonPath(FeedbackFormSessionsData.OLDEST_SESSION_DATE),
                new GdUpdateJsonSettingsUnion()
                        .withNewValue(null)
                        .withJsonPath(FeedbackFormSessionsData.SECOND_SESSION_DATE),
                new GdUpdateJsonSettingsUnion()
                        .withNewValue(timeLong)
                        .withJsonPath(FeedbackFormSessionsData.LAST_SHOW_TIMESTAMP),
                new GdUpdateJsonSettingsUnion()
                        .withNewValue("false")
                        .withJsonPath(FeedbackFormSessionsData.RATE_LATER)
        ));
        var input = new GdSetJsonSettings()
                .withIdType(IdType.UID)
                .withUpdateItems(updateItems);
        setJsonSettings(null, operatorUid, input);
    }

    /**
     * Записать данные о сессиях. Если передан null, то дату не сохраняем
     * @param operatorUid — uid оператора
     * @param oldestDate — дата первой сессии
     * @param secondDate — дата второй сессии
     */
    public void feedbackWriteSessionsDates(Long operatorUid,
            @Nullable LocalDate oldestDate, @Nullable LocalDate secondDate) {
        var updateItems = addRootFeedbackKey(operatorUid);
        if (oldestDate != null) {
            updateItems.add(new GdUpdateJsonSettingsUnion()
                    .withNewValue(oldestDate.toEpochDay())
                    .withJsonPath(FeedbackFormSessionsData.OLDEST_SESSION_DATE));
        }
        if (secondDate != null) {
            updateItems.add(new GdUpdateJsonSettingsUnion()
                    .withNewValue(secondDate.toEpochDay())
                    .withJsonPath(FeedbackFormSessionsData.SECOND_SESSION_DATE));
        }
        var input = new GdSetJsonSettings()
                .withIdType(IdType.UID)
                .withUpdateItems(updateItems);
        setJsonSettings(null, operatorUid, input);
    }

    @SuppressWarnings("rawtypes")
    private ArrayList<GdUpdateJsonSettingsUnion> addRootFeedbackKey(Long operatorUid) {
        var updateItems = new ArrayList<GdUpdateJsonSettingsUnion>();
        var settings = getFeedbackFormSessionsData(operatorUid);
        if ((!Boolean.TRUE.equals(settings.getRateLater())) && settings.getOldestSessionDate() == null
            && settings.getLastShowTime() == null && settings.getSecondSessionDate() == null) {
            // заводим новый ключ у корня
            updateItems.add(new GdUpdateJsonSettingsUnion()
                    .withNewValue(new LinkedHashMap())
                    .withJsonPath(FeedbackFormSessionsData.SESSIONS_DATA_KEY));
        }
        return updateItems;
    }

    /**
     * Прочитать и вернуть данные о сессиях и времени показа ФОС
     * @param operatorUid — uid оператора
     * @return данные о сессиях и времени показа ФОС
     */
    public FeedbackFormSessionsData getFeedbackFormSessionsData(Long operatorUid) {
        var input = new GdGetJsonSettings()
                .withIdType(IdType.UID)
                .withJsonPath(List.of(FeedbackFormSessionsData.SESSIONS_DATA_KEY));
        var settings = getJsonSettings(null, operatorUid, input);
        if (settings.get(0) == null) {
            return new FeedbackFormSessionsData(null, null, null, null);
        }
        var mapper = new ObjectMapper();
        try {
            return mapper.readValue(settings.get(0), FeedbackFormSessionsData.class);
        } catch (Exception ignored) {
            return new FeedbackFormSessionsData(null, null, null, null);
        }
    }
}
