package ru.yandex.direct.grid.core.frontdb.jsonsettings.service.model;

import java.io.IOException;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.ZoneOffset;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;

public class FeedbackFormSessionsData {
    public static final String OLDEST_SESSION_DATE = "$.feedbackSessions.oldestSessionEpochDay";
    public static final String SECOND_SESSION_DATE = "$.feedbackSessions.secondSessionEpochDay";
    public static final String LAST_SHOW_TIMESTAMP = "$.feedbackSessions.lastShowTimestamp";
    public static final String RATE_LATER = "$.feedbackSessions.rateLater";
    public static final String SESSIONS_DATA_KEY = "$.feedbackSessions";

    @JsonProperty("oldestSessionEpochDay")
    @JsonDeserialize(using = DateDeserializer.class)
    private LocalDate oldestSessionDate;

    @JsonProperty("secondSessionEpochDay")
    @JsonDeserialize(using = DateDeserializer.class)
    private LocalDate secondSessionDate;

    @JsonProperty("lastShowTimestamp")
    @JsonDeserialize(using = DateTimeDeserializer.class)
    private LocalDateTime lastShowTime;

    @JsonProperty("rateLater")
    @JsonDeserialize(using = BooleanDeserializer.class)
    private Boolean rateLater;

    public FeedbackFormSessionsData() {
    }

    public FeedbackFormSessionsData(LocalDate oldestSessionDate, LocalDate secondSessionDate, LocalDateTime lastShowTime,
                                    Boolean rateLater) {
        this.oldestSessionDate = oldestSessionDate;
        this.secondSessionDate = secondSessionDate;
        this.lastShowTime = lastShowTime;
        this.rateLater = rateLater;
    }

    public LocalDate getOldestSessionDate() {
        return oldestSessionDate;
    }

    public LocalDate getSecondSessionDate() {
        return secondSessionDate;
    }

    public LocalDateTime getLastShowTime() {
        return lastShowTime;
    }

    public Boolean getRateLater() {
        return rateLater;
    }

    public static class DateDeserializer extends StdDeserializer<LocalDate> {
        public DateDeserializer() {
            this(null);
        }

        public DateDeserializer(Class<?> vc) {
            super(vc);
        }

        @Override
        public LocalDate deserialize(JsonParser p,
                DeserializationContext ctxt) throws IOException
        {
            String date = p.getText();
            if (date == null) {
                return null;
            }
            return LocalDate.ofEpochDay(Long.parseLong(date));
        }
    }

    public static class DateTimeDeserializer extends StdDeserializer<LocalDateTime> {
        public DateTimeDeserializer() {
            this(null);
        }

        public DateTimeDeserializer(Class<?> vc) {
            super(vc);
        }

        @Override
        public LocalDateTime deserialize(JsonParser p,
                DeserializationContext ctxt) throws IOException
        {
            String dateTime = p.getText();
            if (dateTime == null) {
                return null;
            }
            return LocalDateTime.ofEpochSecond(Long.parseLong(dateTime), 0, ZoneOffset.UTC);
        }
    }

    public static class BooleanDeserializer extends StdDeserializer<Boolean> {
        public BooleanDeserializer() {
            this(null);
        }

        public BooleanDeserializer(Class<?> vc) {
            super(vc);
        }

        @Override
        public Boolean deserialize(JsonParser p,
                DeserializationContext ctxt) throws IOException
        {
            String bool = p.getText();
            if (bool == null) {
                return false;
            }
            return "true".equals(bool);
        }
    }


}
