package ru.yandex.direct.grid.core.frontdb.repository;

import java.time.Instant;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.builder.querybuilder.ExtendedWhereBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.FromBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder;
import ru.yandex.direct.ydb.client.DataQueryResultWrapper;
import ru.yandex.direct.ydb.client.ResultSetReaderWrapped;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.table.temptable.TempTableDescription;

import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_CLIENT_BEAN;
import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_PATH_BEAN;
import static ru.yandex.direct.grid.core.frontdb.tables.Tables.FILTER_SHORTCUT_INSTANCES;
import static ru.yandex.direct.utils.HashingUtils.getMd5HashUtf8AsHexString;
import static ru.yandex.direct.ydb.table.temptable.TempTable.tempTable;

@Lazy
@Repository
@ParametersAreNonnullByDefault
public class FilterShortcutRepository {

    private final YdbPath path;
    private final YdbClient ydbClient;

    @Autowired
    public FilterShortcutRepository(@Qualifier(FRONTDB_YDB_CLIENT_BEAN) YdbClient ydbClient,
                                    @Qualifier(FRONTDB_YDB_PATH_BEAN) YdbPath path) {
        this.path = path;
        this.ydbClient = ydbClient;
    }

    public String saveFilter(ClientId clientId, String jsonFilter, @Nullable String hash) {
        if (hash == null) {
            hash = getMd5HashUtf8AsHexString(jsonFilter);
        }

        var tempTable = tempTable(new TempTableDescription(),
                FILTER_SHORTCUT_INSTANCES.HASH,
                FILTER_SHORTCUT_INSTANCES.CLIENT_ID,
                FILTER_SHORTCUT_INSTANCES.SAVE_TIME,
                FILTER_SHORTCUT_INSTANCES.FILTER).createValues();

        tempTable.fill(hash, clientId.asLong(), Instant.now(), jsonFilter);

        FromBuilder queryBuilder = InsertBuilder.replaceInto(FILTER_SHORTCUT_INSTANCES)
                .selectAll()
                .from(tempTable);

        QueryAndParams queryAndParams = queryBuilder.queryAndParams(path);
        ydbClient.executeQuery(queryAndParams);

        return hash;
    }

    @Nullable
    public String getFilter(ClientId clientId, String key) {
        ExtendedWhereBuilder queryBuilder = SelectBuilder.select(FILTER_SHORTCUT_INSTANCES.HASH,
                FILTER_SHORTCUT_INSTANCES.FILTER)
                .from(FILTER_SHORTCUT_INSTANCES)
                .where(FILTER_SHORTCUT_INSTANCES.HASH.eq(key)
                        .and(FILTER_SHORTCUT_INSTANCES.CLIENT_ID.eq(clientId.asLong())));

        QueryAndParams queryAndParams = queryBuilder.queryAndParams(path);

        DataQueryResultWrapper result = ydbClient.executeQuery(queryAndParams);
        ResultSetReaderWrapped resultSet = result.getResultSet(0);

        if (resultSet.next()) {
            return resultSet.getValueReader(FILTER_SHORTCUT_INSTANCES.FILTER).getJsonDocument();
        }

        return null;
    }
}
