package ru.yandex.direct.grid.core.frontdb.repository;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.yandex.ydb.table.SessionRetryContext;
import com.yandex.ydb.table.TableClient;
import one.util.streamex.EntryStream;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import ru.yandex.direct.ydb.YdbPath;
import ru.yandex.direct.ydb.YqlVersion;
import ru.yandex.direct.ydb.builder.QueryAndParams;
import ru.yandex.direct.ydb.builder.expression.Expression;
import ru.yandex.direct.ydb.builder.predicate.Predicate;
import ru.yandex.direct.ydb.builder.querybuilder.ExtendedWhereBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.FromBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.InsertBuilder;
import ru.yandex.direct.ydb.builder.querybuilder.SelectBuilder;
import ru.yandex.direct.ydb.client.YdbClient;
import ru.yandex.direct.ydb.client.YdbSessionProperties;
import ru.yandex.direct.ydb.table.temptable.TempTableDescription;
import ru.yandex.misc.log.mlf.Logger;
import ru.yandex.misc.log.mlf.LoggerFactory;

import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_PATH_BEAN;
import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_SESSION_PROPERTIES_BEAN;
import static ru.yandex.direct.common.configuration.FrontDbYdbConfiguration.FRONTDB_YDB_TABLE_CLIENT_BEAN;
import static ru.yandex.direct.grid.core.frontdb.repository.Tables.CLIENT_SETTINGS;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ydb.builder.expression.JsonExpression.jsonQuery;
import static ru.yandex.direct.ydb.table.temptable.TempTable.tempTable;

@Lazy
@Repository
@ParametersAreNonnullByDefault
public class JsonSettingsRepository {

    private static final String EMPTY_JSON = "{}";
    private static final String EMPTY_JSONDOCUMENT_RESPONSE = "Empty[]";

    private static final Logger logger = LoggerFactory.getLogger(JsonSettingsRepository.class);
    private static final Duration QUERY_TIMEOUT = Duration.ofMinutes(1);

    private final YdbClient ydbClient;
    private final YdbPath path;

    @Autowired
    public JsonSettingsRepository(@Qualifier(FRONTDB_YDB_TABLE_CLIENT_BEAN) TableClient tableClient,
                                  @Qualifier(FRONTDB_YDB_PATH_BEAN) YdbPath path,
                                  @Qualifier(FRONTDB_YDB_SESSION_PROPERTIES_BEAN) YdbSessionProperties ydbSessionProperties) {

        var sessionRetryContext = SessionRetryContext.create(tableClient)
                .maxRetries(ydbSessionProperties.getMaxQueryRetries())
                .retryNotFound(ydbSessionProperties.isRetryNotFound())
                .build();

        this.ydbClient = new YdbClient(sessionRetryContext, QUERY_TIMEOUT);
        this.path = path;
    }

    public List<String> getJsonSettings(@Nullable Long clientId, @Nullable Long operatorUid,
                                        List<String> jsonPathList) {
        //noinspection rawtypes
        var selectExpr = new ArrayList<Expression>();
        for (String jsonPath : jsonPathList) {
            selectExpr.add(jsonQuery(CLIENT_SETTINGS.SETTINGS.getColumnName(), jsonPath));
        }
        Predicate operatorUidPredicate = operatorUid != null ?
                CLIENT_SETTINGS.UID.eq(operatorUid) :
                CLIENT_SETTINGS.UID.isNull();
        Predicate clientIdPredicate = clientId != null ?
                CLIENT_SETTINGS.CLIENT_ID.eq(clientId) :
                CLIENT_SETTINGS.CLIENT_ID.isNull();
        ExtendedWhereBuilder from = SelectBuilder.select(selectExpr)
                .from(CLIENT_SETTINGS)
                .where(operatorUidPredicate.and(clientIdPredicate));

        QueryAndParams queryAndParams = from.queryAndParams(path, YqlVersion.SQLv1);

        var resultSet = ydbClient.executeQuery(queryAndParams,
                "Failed to get settings", true).getResultSet(0);

        if (resultSet.next()) {
            return EntryStream.of(jsonPathList)
                    .mapKeyValue((index, path) -> resultSet.getColumn(index).toString())
                    .map(JsonSettingsRepository::extractSettingFromRawValue)
                    .toList();
        } else {
            return mapList(jsonPathList, path -> EMPTY_JSON);
        }
    }

    private static String extractSettingFromRawValue(String r) {
        //примеры валидных ответов: Some["{"123":123}"] и Empty[]
        if (r.startsWith("Some[")) {
            return r.substring(6, r.length() - 2);
        } else if (r.equals(EMPTY_JSONDOCUMENT_RESPONSE)) {
            return EMPTY_JSON;
        } else {
            throw new IllegalStateException("Unknown jsonDocument value: " + r);
        }
    }

    public void updateJsonSettings(@Nullable Long clientId, @Nullable Long operatorUid, String jsonPath) {
        logger.info("Start changing client settings");
        var tempTable = tempTable(new TempTableDescription(), CLIENT_SETTINGS.CLIENT_ID,
                CLIENT_SETTINGS.UID,
                CLIENT_SETTINGS.SETTINGS).createValues();

        tempTable.fill(clientId, operatorUid, jsonPath);

        FromBuilder fromBuilder = InsertBuilder.replaceInto(CLIENT_SETTINGS)
                .selectAll()
                .from(tempTable);

        QueryAndParams queryAndParams1 = fromBuilder.queryAndParams(path, YqlVersion.SQLv1);
        ydbClient.executeQuery(queryAndParams1);
    }

}
