package ru.yandex.direct.grid.core.util.filters;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import static ru.yandex.direct.utils.FunctionalUtils.filterList;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Хелпер для фильтрации списка однородных объектов по классу, содержащему описание применяемых в текущий момент фильтров.
 * Также может выступать как предикат, например в фильтрации стримов
 * <p>
 * Хелпер всегда считает объекты подходящими, если значение фильтра - null, и всегда отбрасывает элемент, если его
 * значение null, а для него есть непустые фильтры
 *
 * @param <F> тип класса с описанием фильтров
 * @param <L> тип элемента фильтруемого списка
 */
@ParametersAreNonnullByDefault
public class FilterProcessor<F, L> implements BiPredicate<F, L> {
    @ParametersAreNonnullByDefault
    private static class FilterItem<F, L, T> implements BiPredicate<F, L> {
        private final Function<F, T> valueExtractor;
        private final BiPredicate<T, L> predicate;

        private FilterItem(Function<F, T> valueExtractor, BiPredicate<T, L> predicate) {
            this.valueExtractor = valueExtractor;
            this.predicate = predicate;
        }

        @Override
        public boolean test(F filtersContainer, L listItem) {
            T filterValue = valueExtractor.apply(filtersContainer);
            return filterValue == null || listItem != null && predicate.test(filterValue, listItem);
        }
    }

    /**
     * Билдер для хелпера фильтрации
     *
     * @param <F> тип класса с описанием фильтров
     * @param <L> тип элемента фильтруемого списка
     */
    @ParametersAreNonnullByDefault
    public static class Builder<F, L> {
        private final List<BiPredicate<F, L>> filters = new ArrayList<>();

        /**
         * Добавить фильтр
         *
         * @param filterExtractor функция для извлечения проверяемого поля из объекта с описанием применяемых фильтров
         * @param predicate       предикат для проверки соответствия элемента списка извлеченному значению фильтра
         * @param <T>             тип значения фильтра
         */
        public <T> Builder<F, L> withFilter(Function<F, T> filterExtractor, BiPredicate<T, L> predicate) {
            filters.add(new FilterItem<>(filterExtractor, predicate));
            return this;
        }

        private <T> BiPredicate<F, T> preExtract(Function<T, L> extractor, BiPredicate<F, L> predicate) {
            return (f, t) -> {
                L itemValue = extractor.apply(t);
                return itemValue != null && predicate.test(f, itemValue);
            };
        }

        public <T> FilterProcessor<F, T> build(Function<T, L> extractor) {
            List<BiPredicate<F, T>> newFilters = mapList(filters, f -> preExtract(extractor, f));
            return new FilterProcessor<>(newFilters);
        }

        public FilterProcessor<F, L> build() {
            return new FilterProcessor<>(filters);
        }
    }

    private final List<BiPredicate<F, L>> filters;

    private FilterProcessor(List<BiPredicate<F, L>> filters) {
        this.filters = new ArrayList<>(filters);
    }

    /**
     * Вернуть список элементов коллекции, выполняющих условия фильтрации, описанные в переданном объекте
     *
     * @param filtersContainer объект с описанием применяемых условий фильтрации
     * @param items            коллекция, элементы которой нужно отфильтровать
     */
    public List<L> filter(F filtersContainer, Collection<L> items) {
        return filterList(items, predicate(filtersContainer));
    }

    /**
     * Вернуть предикат, возвразающий истину при выполнении условий фильтрации, описанных в переданном объекте
     *
     * @param filtersContainer объект с описанием применяемых условий фильтрации
     */
    public Predicate<L> predicate(F filtersContainer) {
        return l -> test(filtersContainer, l);
    }

    /**
     * Вернуть истину если проверяемый элемент соответствует условиям фильтрации, описанным в переданном объекте
     *
     * @param filtersContainer объект с описанием применяемых условий фильтрации
     * @param item             проверяемый элемент
     */
    @Override
    public boolean test(F filtersContainer, L item) {
        if (filtersContainer == null) {
            return true;
        }
        for (BiPredicate<F, L> predicate : filters) {
            if (!predicate.test(filtersContainer, item)) {
                return false;
            }
        }
        return true;
    }
}
