package ru.yandex.direct.grid.core.util.filters;

import java.time.LocalDate;
import java.util.Collection;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;

import javax.annotation.ParametersAreNonnullByDefault;

import static ru.yandex.direct.validation.BiPredicates.not;

/**
 * Предикаты для упрощения создания {@link FilterProcessor}
 */
@ParametersAreNonnullByDefault
public class FilterProvider {
    private FilterProvider() {
    }

    /**
     * Предикат для проверки на то, что список допустимых значений, заданный в фильтре, содержит значение поля
     * объекта проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <F, L> BiPredicate<Set<F>, L> contains(Function<L, F> function) {
        return (f, l) -> f.contains(function.apply(l));
    }

    /**
     * Предикат для проверки на то, что список допустимых значений, заданный в фильтре, содержит значение поля
     * объекта проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <F, L> BiPredicate<Set<F>, L> containsMatching(Function<L, F> function, BiPredicate<F, F> predicate) {
        return (f, l) -> f.stream().anyMatch(fi -> testNullable(function, predicate).test(fi, l));
    }

    /**
     * Предикат для проверки на то, что каждому списку допустимых значений, заданных в фильтре, есть хотя бы одно
     * соответствие в поле объекта проверяемого класса.
     * Позволяет реализовать, например, такую проверку соответствия значения:
     * (условие1 ИЛИ условие2) И (условие3 ИЛИ условие4 ИЛИ условие5) И (условие6)
     *
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <F, L> BiPredicate<Set<Set<F>>, L> innerContainsMatching(BiPredicate<F, L> predicate) {
        return (s, l) -> s.stream()
                .allMatch(f ->
                        f.stream()
                                .anyMatch(fi -> predicate.test(fi, l))
                );
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, идентично значению поля объекта проверяемого
     * класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <F, L> BiPredicate<F, L> equalTo(Function<L, F> function) {
        return (f, l) -> f.equals(function.apply(l));
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, является подстрокой значения поля объекта
     * проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<String, L> isSubString(Function<L, String> function) {
        return testNullable(function, (f, v) -> v.contains(f));
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, является подстрокой значения поля объекта
     * проверяемого класса. Без учета регистра символов
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<String, L> isSubStringCU(Function<L, String> function) {
        return testNullable(function, (f, v) -> v.toLowerCase().contains(f.toLowerCase()));
    }

    /**
     * Предикат для проверки на то, что какому-то элементу списка допустимых подстрок, заданных в фильтре, есть
     * соответствие в числовом поле объекта проверяемого класса.
     * Позволяет найти объекты, которые, например, содержат "025" или "100" в id
     *
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<Collection<String>, L> numericContainsAny(Function<L, Number> function) {
        return (f, l) -> {
            var numberString = function.apply(l).toString();
            return f.stream().anyMatch(numberString::contains);
        };
    }

    /**
     * Предикат для проверки на то, что значение даты, заданное в фильтре, на временной шкале находится ранее чем
     * значения поля объекта проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<LocalDate, L> before(Function<L, LocalDate> function) {
        return testNullable(function, LocalDate::isBefore);
    }

    /**
     * Предикат для проверки на то, что значение даты, заданное в фильтре, на временной шкале находится ранее или равно
     * значению поля объекта проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<LocalDate, L> beforeOrEqual(Function<L, LocalDate> function) {
        return testNullable(function, not(LocalDate::isAfter));
    }

    /**
     * Предикат для проверки на то, что значение даты, заданное в фильтре, на временной шкале находится позже чем
     * значения поля объекта проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<LocalDate, L> after(Function<L, LocalDate> function) {
        return testNullable(function, LocalDate::isAfter);
    }

    /**
     * Предикат для проверки на то, что значение даты, заданное в фильтре, на временной шкале находится позже или равно
     * значению поля объекта проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <L>      проверяемый класс
     */
    public static <L> BiPredicate<LocalDate, L> afterOrEqual(Function<L, LocalDate> function) {
        return testNullable(function, not(LocalDate::isBefore));
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, меньше чем значение поля объекта проверяемого
     * класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <L, F extends Comparable<F>> BiPredicate<F, L> lessThan(Function<L, F> function) {
        return testNullable(function, (f, v) -> f.compareTo(v) < 0);
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, меньше или равно значению поля объекта
     * проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <L, F extends Comparable<F>> BiPredicate<F, L> lessOrEqual(Function<L, F> function) {
        return testNullable(function, (f, v) -> f.compareTo(v) <= 0);
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, больше чем значение поля объекта проверяемого
     * класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <L, F extends Comparable<F>> BiPredicate<F, L> greaterThan(Function<L, F> function) {
        return testNullable(function, (f, v) -> f.compareTo(v) > 0);
    }

    /**
     * Предикат для проверки на то, что значение, заданное в фильтре, больше или равно значению поля объекта
     * проверяемого класса
     *
     * @param function функция для получения значения поля проверяемого объекта
     * @param <F>      класс допустимых значений, заданных в фильтре
     * @param <L>      проверяемый класс
     */
    public static <L, F extends Comparable<F>> BiPredicate<F, L> greaterOrEqual(Function<L, F> function) {
        return testNullable(function, (f, v) -> f.compareTo(v) >= 0);
    }

    private static <L, F> BiPredicate<F, L> testNullable(Function<L, F> function, BiPredicate<F, F> predicate) {
        return (f, l) -> {
            F val = function.apply(l);
            return val != null && predicate.test(f, val);
        };
    }
}
