package ru.yandex.direct.grid.core.util.filters;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.function.Function;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Condition;

/**
 * Хелпер для создания jooq-условий фильтрации результатов запроса к базе данных по классу, содержащему описание
 * применяемых в текущий момент фильтров.
 *
 * @param <F> тип класса с описанием фильтров
 */
@ParametersAreNonnullByDefault
public class JooqFilterProcessor<F> implements Function<F, Condition> {
    @ParametersAreNonnullByDefault
    private static class ConditionProvider<F, T> implements Function<F, Condition> {
        private final Function<F, T> filterExtractor;
        private final Function<T, Condition> conditionBuilder;

        private ConditionProvider(Function<F, T> filterExtractor, Function<T, Condition> conditionBuilder) {
            this.filterExtractor = filterExtractor;
            this.conditionBuilder = conditionBuilder;
        }

        @Nullable
        @Override
        public Condition apply(F filterDescription) {
            T filterValue = filterExtractor.apply(filterDescription);
            return filterValue != null ? conditionBuilder.apply(filterValue) : null;
        }
    }

    @ParametersAreNonnullByDefault
    public static class Builder<F> {
        private final List<Function<F, Condition>> conditionProviders = new ArrayList<>();

        /**
         * Добавить фильтра
         *
         * @param filterExtractor  функция для получения значения фильтра из контеинера
         * @param conditionBuilder функция для создания условия из полученного значения, если оно не null. Если полученное
         *                         значение null - фильтр пропускается
         */
        public <T> Builder<F> withFilter(Function<F, T> filterExtractor, Function<T, Condition> conditionBuilder) {
            conditionProviders.add(new ConditionProvider<>(filterExtractor, conditionBuilder));
            return this;
        }

        public JooqFilterProcessor<F> build() {
            return new JooqFilterProcessor<>(new ArrayList<>(conditionProviders));
        }
    }

    private final List<Function<F, Condition>> conditionProviders;

    private JooqFilterProcessor(List<Function<F, Condition>> conditionProviders) {
        this.conditionProviders = conditionProviders;
    }

    public static <F> Builder<F> builder() {
        return new Builder<>();
    }

    /**
     * Создать jooq-условие по переданному описанию фильтров
     */
    @Nullable
    @Override
    public Condition apply(@Nullable F filterDescription) {
        if (filterDescription == null) {
            return null;
        }
        return conditionProviders.stream()
                .map(c -> c.apply(filterDescription))
                .filter(Objects::nonNull)
                .reduce(Condition::and)
                .orElse(null);
    }
}
