package ru.yandex.direct.grid.core.util.filters;

import java.math.BigDecimal;
import java.util.Set;
import java.util.function.Function;

import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.TableField;
import org.jooq.impl.DSL;

import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;

import static org.jooq.impl.DSL.field;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Функции хелперы для построения фильтрующих запросов с помощью JooqFilterProcessor
 */
public class JooqFilterProvider {
    private JooqFilterProvider() {
    }

    /**
     * Создать функцию, возвращающую условие-проверку на то, что переданное созданной функции значение меньше или равно
     * переданному полю
     *
     * @param field поле, для которого нужно создать условие
     */
    public static <N extends Number> Function<N, Condition> lessOrEqual(Field<BigDecimal> field) {
        return v -> field.ge(field(v.toString(), BigDecimal.class));
    }

    /**
     * Создать функцию, возвращающую условие-проверку на то, что переданное созданной функции значение меньше или равно
     * переданному полю
     *
     * @param field      поле, для которого нужно создать условие
     * @param multiplier число, на которое нужно умножить переданное в проверку значение перед сравнением
     */
    public static Function<BigDecimal, Condition> lessOrEqual(Field<BigDecimal> field, BigDecimal multiplier) {
        return v -> lessOrEqual(field).apply(multiplier.multiply(v).longValue());
    }

    /**
     * Создать функцию, возвращающую условие-проверку на то, что переданное созданной функции значение больше или равно
     * переданному полю
     *
     * @param field поле, для которого нужно создать условие
     */
    public static <N extends Number> Function<N, Condition> greaterOrEqual(Field<BigDecimal> field) {
        return v -> field.le(field(v.toString(), BigDecimal.class));
    }

    /**
     * Создать функцию, возвращающую условие-проверку на то, что переданное созданной функции значение больше или равно
     * переданному полю
     *
     * @param field      поле, для которого нужно создать условие
     * @param multiplier число, на которое нужно умножить переданное в проверку значение перед сравнением
     */
    public static Function<BigDecimal, Condition> greaterOrEqual(Field<BigDecimal> field, BigDecimal multiplier) {
        return v -> greaterOrEqual(field).apply(multiplier.multiply(v).longValue());
    }

    /**
     * Проверка на то, что значение переданного поля содержится в наборе переданных ожидаемых значений. Проверка идет
     * без учета регистра символов
     */
    public static Function<Set<String>, Condition> inSetIgnoreCase(TableField<?, String> field) {
        return s -> YtDSL.toLower(field).in(mapList(s, String::toLowerCase));
    }

    /**
     * Проверка на то, что значение переданного поля содержит подстроку с ожидаемым значенем. Проверка идет
     * без учета регистра символов
     */
    public static Function<String, Condition> substringIgnoreCase(Field<String> field) {
        return s -> YtDSL.isSubstring(s.toLowerCase(), YtDSL.toLower(field));
    }

    /**
     * Проверка на то, что значение переданного поля содержит подстроку с ожидаемым значением.
     * Проверка идет без учета регистра и типа пробелов
     *
     * @param field поле из базы
     * @return функцию возвращающую условие
     */
    public static Function<String, Condition> substringIgnoreCaseAndWhitespaceType(TableField<?, String> field) {
        return s -> YtDSL.isSubstringIgnoringWhitespaceType(s.toLowerCase(), YtDSL.toLower(field));
    }

    /**
     * Проверка на то, что значение переданного поля содержит подстроку со значением одного из элементов переданного
     * набора. Проверка идет без учета регистра символов
     */
    public static Function<Set<String>, Condition> inSetSubstringIgnoreCase(Field<String> field) {
        return s -> {
            var transformedField = YtDSL.toLower(field);
            return s.stream().reduce(
                    DSL.noCondition(),
                    (c, e) -> c.or(YtDSL.isSubstring(e.toLowerCase(), transformedField)),
                    Condition::or
            );
        };
    }

    /**
     * Проверка на то, что значение переданного поля содержит подстроку со значением одного из элементов в каждом
     * переданном наборе. Проверка идет без учета регистра символов
     */
    public static Function<Set<Set<String>>, Condition> inAllSetsOfSetSubstringIgnoreCase(Field<String> field) {
        return sets -> {
            var transformedField = YtDSL.toLower(field);
            return sets.stream().reduce(
                    DSL.noCondition(),
                    (andCondition, substrings) -> andCondition.and(
                            substrings.stream().reduce(
                                    DSL.noCondition(),
                                    (orCondition, substring) -> orCondition.or(
                                            YtDSL.isSubstring(substring.toLowerCase(), transformedField)),
                                    Condition::or
                            )
                    ),
                    Condition::and
            );
        };
    }

    /**
     * Проверка на то, что значение переданного числового поля содержит подстроку со значением одного из элементов
     * переданного набора. Числа в наборе должны быть обращены в строки заранее.
     */
    public static Function<Set<String>, Condition> inSetNumericSubstring(Field<Long> field) {
        return s -> {
            var transformedField = YtDSL.numericToString(field);
            return s.stream().reduce(
                    DSL.noCondition(),
                    (c, e) -> c.or(YtDSL.isSubstring(e, transformedField)),
                    Condition::or
            );
        };
    }

    /**
     * Отрицание переданного условия
     */
    public static <F> Function<F, Condition> not(Function<F, Condition> base) {
        return s -> DSL.not(base.apply(s));
    }

    /**
     * Объединение трёх функций по условию
     *
     * @param base1 функция по условию 1
     * @param base2 функция по условию 2
     * @param base3 функция по условию 3
     * @param <F>   тип первого аргумента объединяемых функций
     * @return функция, возвращающая правду, если бы правду возвращала хотя бы одна из изначальных
     */
    public static <F> Function<F, Condition> or(Function<F, Condition> base1, Function<F, Condition> base2,
            Function<F, Condition> base3) {
        return s -> DSL.or(base1.apply(s), base2.apply(s), base3.apply(s));
    }
}
