package ru.yandex.direct.grid.core.util.money;

import java.math.BigDecimal;
import java.math.MathContext;
import java.math.RoundingMode;
import java.util.function.Consumer;
import java.util.function.Supplier;

import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.client.model.ClientNds;
import ru.yandex.direct.currency.Percent;

import static ru.yandex.direct.currency.MoneyUtils.INTERNAL_MONEY_SCALE;
import static ru.yandex.direct.currency.MoneyUtils.subtractPercentPrecisely;

/**
 * Методы для работы с суммами с повышенной точностью
 * Сейчас нам требуется большая точность чем предоставляет Money.subtractNds.
 * Возможно в будущем научимся пользоваться одним только Money. Обуждаем здесь (DIRECT-82781)
 */
@ParametersAreNonnullByDefault
public class GridMoneyUtils {

    /**
     * Вычитает НДС из суммы, предоставляемой {@code supplier}'ом. Сохраняет результат используя указанный {@code
     * consumer}
     *
     * @param supplier  геттер поля
     * @param consumer  сеттер поля
     * @param clientNds НДС клиента
     */
    public static void applyNdsSubtruction(Supplier<BigDecimal> supplier, Consumer<BigDecimal> consumer,
                                           ClientNds clientNds) {
        BigDecimal sum = supplier.get();
        if (sum == null) {
            return;
        }
        consumer.accept(subtractPercentPrecisely(sum, clientNds.getNds()));
    }

    /**
     * Добавляет НДС к сумме, предоставляемой {@code supplier}'ом. Сохраняет результат используя указанный {@code
     * consumer}
     *
     * @param supplier  геттер поля
     * @param consumer  сеттер поля
     * @param clientNds НДС клиента
     */
    public static void applyNdsAddition(Supplier<BigDecimal> supplier, Consumer<BigDecimal> consumer,
                                        ClientNds clientNds) {
        applyNdsAddition(supplier, consumer, clientNds.getNds());
    }

    /**
     * Добавляет НДС к сумме, предоставляемой {@code supplier}'ом. Сохраняет результат используя указанный {@code
     * consumer}
     *
     * @param supplier         геттер поля
     * @param consumer         сеттер поля
     * @param clientNdsPercent НДС клиента
     */
    public static void applyNdsAddition(Supplier<BigDecimal> supplier, Consumer<BigDecimal> consumer,
                                        Percent clientNdsPercent) {
        BigDecimal sum = supplier.get();
        if (sum == null) {
            return;
        }
        consumer.accept(addPercentPrecisely(sum, clientNdsPercent));
    }

    /**
     * Добавляет процент к заданной сумме
     *
     * @param sum     - сумма, из которой нужно вычесть проценты
     * @param percent - вычитаемый процент
     */
    public static BigDecimal addPercentPrecisely(BigDecimal sum, Percent percent) {
        return sum.multiply(BigDecimal.ONE.add(percent.asRatio()), MathContext.DECIMAL64)
                .setScale(INTERNAL_MONEY_SCALE, RoundingMode.DOWN);
    }
}
