package ru.yandex.direct.grid.core.util.ordering;

import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.OrderField;

import ru.yandex.direct.grid.model.Order;

/**
 * Хелпер для создания списка полей, по которым нужно сортировать результаты запроса в базу данных
 *
 * @param <O> тип класса, описывающего поле, по которому будет производиться сортировка
 */
@ParametersAreNonnullByDefault
public class JooqOrderingProcessor<O> {
    private final Map<O, Field<?>> objToField;

    private JooqOrderingProcessor(Map<O, Field<?>> objToField) {
        this.objToField = objToField;
    }

    @Nullable
    private OrderField<?> constructField(OrderingItem<O> orderingItem) {
        Field<?> field = objToField.get(orderingItem.getField());
        if (field == null) {
            throw new IllegalArgumentException(String.format("Unknown ordering item %s", orderingItem.getField()));
        }

        return orderingItem.getOrder() == Order.DESC ? field.desc() : field;
    }

    /**
     * Получить список полей по списку их описаний
     */
    public List<OrderField<?>> construct(@Nullable List<? extends OrderingItem<O>> orderBy) {
        if (orderBy == null) {
            return Collections.emptyList();
        }
        return orderBy.stream()
                .map(this::constructField)
                .filter(Objects::nonNull)
                .collect(Collectors.toList());
    }

    public static <O> Builder<O> builder() {
        return new Builder<>();
    }

    public static class Builder<O> {
        private final Map<O, Field<?>> objToField = new HashMap<>();

        public Builder<O> withField(O obj, Field<?> field) {
            objToField.put(obj, field);
            return this;
        }

        public JooqOrderingProcessor<O> build() {
            return new JooqOrderingProcessor<>(new HashMap<>(objToField));
        }
    }
}
