package ru.yandex.direct.grid.core.util.ordering;

import java.util.Collection;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;

import ru.yandex.direct.grid.model.Order;

/**
 * Хелпер для обработки запросов на сортировку сложных объектов по их полям
 *
 * @param <O> тип класса, описывающего поле, по которому будет производиться сортировка
 * @param <I> тип сортируемых объектов
 */
@ParametersAreNonnullByDefault
public class OrderingProcessor<O, I> {
    private final Map<O, Comparator<I>> objToComparator;

    private OrderingProcessor(Map<O, Comparator<I>> objToComparator) {
        this.objToComparator = objToComparator;
    }

    /**
     * Вернуть компаратор для сортировки в соответствии с переданными условиями
     *
     * @param orderBy условия сортировки
     */
    public Comparator<I> comparator(@Nullable List<? extends OrderingItem<O>> orderBy) {
        if (CollectionUtils.isEmpty(orderBy)) {
            return (a, b) -> 0;
        }

        Comparator<I> comparator = null;
        for (OrderingItem<O> item : orderBy) {
            Comparator<I> currentComparator = objToComparator.get(item.getField());
            if (currentComparator == null) {
                throw new IllegalArgumentException(String.format("No comparator for %s", item.getField()));
            }

            currentComparator = item.getOrder() == Order.ASC ? currentComparator : currentComparator.reversed();

            if (comparator == null) {
                comparator = currentComparator;
            } else {
                comparator = comparator.thenComparing(currentComparator);
            }
        }
        if (comparator == null) {
            throw new IllegalArgumentException("Empty list of comparators");
        }

        return comparator;
    }

    /**
     * Вернуть список элементов переданной коллекции, отсортированный с соответствии с заданным условием
     *
     * @param items   коллекция для сортировки
     * @param orderBy условия сортировки
     */
    public List<I> sort(Collection<I> items, @Nullable List<OrderingItem<O>> orderBy) {
        return items.stream()
                .sorted(comparator(orderBy))
                .collect(Collectors.toList());
    }

    /**
     * Создать билдер хелпера
     *
     * @param nullFirst true, если null-значения нужно перемещать в начало отсортированного списка
     */
    public static <O, I> Builder<O, I> builder(boolean nullFirst) {
        return new Builder<>(nullFirst);
    }

    @ParametersAreNonnullByDefault
    public static class Builder<O, I> {
        private final Map<O, Comparator<I>> fieldToComparator;
        private final boolean nullFirst;

        private Builder(boolean nullFirst) {
            this.nullFirst = nullFirst;
            fieldToComparator = new HashMap<>();
        }

        /**
         * Добавить сортировку по заданному полю с переданным компаратором
         *
         * @param field      объект описания поля
         * @param comparator компаратор поля
         */
        public Builder<O, I> withComparator(O field, Comparator<I> comparator) {
            fieldToComparator
                    .put(field, nullFirst ? Comparator.nullsFirst(comparator) : Comparator.nullsLast(comparator));
            return this;
        }

        /**
         * Добавить сортировку по заданному полю с переданным компаратором
         *
         * @param field           объект описания поля
         * @param extractor       экстрактор значения поля
         * @param fieldComparator компаратор поля
         */
        public <F> Builder<O, I> withFieldComparator(O field, Function<I, F> extractor, Comparator<F> fieldComparator) {
            Comparator<F> newComparator =
                    nullFirst ? Comparator.nullsFirst(fieldComparator) : Comparator.nullsLast(fieldComparator);
            Comparator<I> comparator = (a, b) -> newComparator.compare(extractor.apply(a), extractor.apply(b));
            return withComparator(field, comparator);
        }

        /**
         * Добавить сортировку по заданному полю с компаратором Comparator.naturalOrder()
         *
         * @param field     объект описания поля
         * @param extractor экстрактор значения поля
         */
        public <F extends Comparable<F>> Builder<O, I> withFieldComparator(O field, Function<I, F> extractor) {
            return withFieldComparator(field, extractor, Comparator.naturalOrder());
        }

        public OrderingProcessor<O, I> build() {
            return new OrderingProcessor<>(fieldToComparator);
        }
    }
}
