package ru.yandex.direct.grid.core.util.stats;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableMap;
import org.apache.commons.collections4.CollectionUtils;
import org.jooq.Condition;
import org.jooq.Field;
import org.jooq.OrderField;
import org.jooq.Record;
import org.jooq.Select;
import org.jooq.SelectHavingStep;
import org.jooq.SelectJoinStep;
import org.jooq.SelectSelectStep;
import org.jooq.Table;
import org.jooq.TableOnConditionStep;
import org.jooq.impl.DSL;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStatsFilter;
import ru.yandex.direct.grid.core.entity.model.GdiGoalConversion;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStats;
import ru.yandex.direct.grid.core.entity.model.GdiGoalStatsFilter;
import ru.yandex.direct.grid.core.entity.model.GdiStatsOrderBySupportedField;
import ru.yandex.direct.grid.core.util.filters.JooqFilterProcessor;
import ru.yandex.direct.grid.core.util.ordering.JooqOrderingProcessor;
import ru.yandex.direct.grid.core.util.ordering.OrderingItem;
import ru.yandex.direct.grid.core.util.ordering.OrderingItemExtended;
import ru.yandex.direct.grid.core.util.stats.completestat.GridStatTableData;
import ru.yandex.direct.grid.core.util.stats.goalstat.DirectGoalPhraseStatData;
import ru.yandex.direct.grid.core.util.stats.goalstat.GridGoalStatTableData;
import ru.yandex.direct.grid.model.Order;
import ru.yandex.direct.grid.schema.yt.tables.DirectphrasegoalsstatBs;
import ru.yandex.direct.grid.schema.yt.tables.Directphrasestatv2Bs;
import ru.yandex.direct.intapi.client.model.request.statistics.option.ReportOptionGroupByDate;
import ru.yandex.direct.ytwrapper.YtUtils;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtMappingUtils;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static java.math.BigDecimal.ZERO;
import static java.util.Arrays.asList;
import static java.util.Collections.emptySet;
import static org.jooq.impl.DSL.field;
import static org.jooq.impl.DSL.row;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.greaterOrEqual;
import static ru.yandex.direct.grid.core.util.filters.JooqFilterProvider.lessOrEqual;
import static ru.yandex.direct.grid.core.util.stats.GridStatUtils.DEFAULT_AUTOBUDGET_STRATEGY_ID;
import static ru.yandex.direct.grid.schema.yt.Tables.DIRECTGRIDGOALSSTAT_BS;
import static ru.yandex.direct.grid.schema.yt.Tables.DIRECTPHRASEGOALSSTAT_BS;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.ytFalse;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.ytIfNull;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL.ytTrue;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.DECIMAL_MULT;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.PERCENT_DECIMAL_MULT;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.extractBsMoney;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.extractBsPercent;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.longNodeBigDecimal;
import static ru.yandex.direct.ytwrapper.dynamic.dsl.YtQueryUtil.longNodeLocalDate;

/**
 * Класс, содержащий вспомогательные объекты и методы для построения запросов к выгружаемой БК статистике
 */
@ParametersAreNonnullByDefault
public class GridStatNew<T extends Table, T1 extends Table> {
    private static final Logger logger = LoggerFactory.getLogger(GridStatNew.class);

    private static final int GOAL_COUNT_THRESHOLD_FOR_LOGGING = 50;
    public static final long STRATEGY_GOAL_TYPE = 1L;
    public static final long MEANINGFUL_GOAL_TYPE = 2L;
    private static final long THREE_HOUR = 10800L;
    private static final long INSTALL_CLICK_GOAL_ID = 4L;

    private static final String GOAL_STAT_ALIAS = "GS";
    private final Field<BigDecimal> cost;
    private final Field<BigDecimal> costWithTax;
    private final Field<BigDecimal> revenue;
    private final Field<BigDecimal> shows;
    private final Field<BigDecimal> clicks;
    private final Field<BigDecimal> goalsNum;
    private final Field<BigDecimal> cpmPrice;
    private final Field<BigDecimal> firstPageShows;
    private final Field<BigDecimal> firstPageClicks;
    private final Field<BigDecimal> firstPageSumPosShows;
    private final Field<BigDecimal> firstPageSumPosClicks;
    // В РМП кампаниях мы считаем количество сессий как количество кликов. Потому что в МОЛ так. Такие дела
    private final Field<BigDecimal> sessionNum;
    private final Field<BigDecimal> sessionNumLimited;
    private final Field<BigDecimal> bounces;
    private final Field<BigDecimal> sessionDepth;

    private final Field<BigDecimal> ctr;
    private final Field<BigDecimal> avgClickCost;
    private final Field<BigDecimal> avgShowPosition;
    private final Field<BigDecimal> avgClickPosition;
    private final Field<BigDecimal> bounceRate;
    private final Field<BigDecimal> avgDepth;
    private final Field<BigDecimal> conversionRate;
    private final Field<BigDecimal> avgGoalCost;
    private final Field<BigDecimal> profitability;
    private final Field<BigDecimal> crr;

    private final GridStatTableData<T, T1> tableData;

    private final JooqFilterProcessor<GdiEntityStatsFilter> statFilterProcessor;
    private final Map<String, Field<?>> statOrderFields;

    public GridStatNew(GridStatTableData<T, T1> tableData) {
        this.tableData = tableData;

        cost = YtDSL.ytIfNull(
                DSL.sum(YtDSL.ytIf(tableData.currencyId().eq(0L),
                        // Сумма в фишках всегда с НДС 1.18. Вычитаем его
                        tableData.cost().multiply(100).divide(118),
                        tableData.costTaxFree())),
                BigDecimal.ZERO)
                .as(GdiEntityStats.COST.name());
        costWithTax = YtDSL.ytIfNull(DSL.sum(tableData.costCur()), BigDecimal.ZERO)
                .as(GdiEntityStats.COST_WITH_TAX.name());
        revenue = YtDSL.ytIfNull(DSL.sum(tableData.priceCur()), BigDecimal.ZERO)
                .as(GdiEntityStats.REVENUE.name());
        shows = YtDSL.ytIfNull(DSL.sum(tableData.shows()), BigDecimal.ZERO)
                .as(GdiEntityStats.SHOWS.name());
        clicks = YtDSL.ytIfNull(DSL.sum(tableData.clicks()), BigDecimal.ZERO)
                .as(GdiEntityStats.CLICKS.name());
        goalsNum = YtDSL.ytIfNull(DSL.sum(tableData.goalsNum()), BigDecimal.ZERO)
                .as(GdiEntityStats.GOALS.name());
        cpmPrice = YtDSL.ytIf(shows.gt(BigDecimal.ZERO),
                cost.divide(shows).multiply(1000),
                DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.CPM_PRICE.name());
        firstPageShows = YtDSL.ytIfNull(DSL.sum(tableData.firstPageShows()), BigDecimal.ZERO)
                .as(GdiEntityStats.FIRST_PAGE_SHOWS.name());
        firstPageClicks = YtDSL.ytIfNull(DSL.sum(tableData.firstPageClicks()), BigDecimal.ZERO)
                .as(GdiEntityStats.FIRST_PAGE_CLICKS.name());
        firstPageSumPosShows = YtDSL.ytIfNull(DSL.sum(tableData.firstPageSumPosShows()), BigDecimal.ZERO)
                .as(GdiEntityStats.FIRST_PAGE_SUM_POS_SHOWS.name());
        firstPageSumPosClicks = YtDSL.ytIfNull(DSL.sum(tableData.firstPageSumPosClicks()), BigDecimal.ZERO)
                .as(GdiEntityStats.FIRST_PAGE_SUM_POS_CLICKS.name());
        // В РМП кампаниях мы считаем количество сессий как количество кликов. Потому что в МОЛ так. Такие дела
        sessionNum = YtDSL.ytIfNull(
                DSL.sum(YtDSL.ytIf(YtDSL.isTrue(tableData.isRmp()), tableData.clicks(), tableData.sessionNum())),
                BigDecimal.ZERO)
                .as(GdiEntityStats.SESSIONS.name());
        sessionNumLimited = YtDSL.ytIfNull(DSL.sum(tableData.sessionNumLimited()), BigDecimal.ZERO)
                .as(GdiEntityStats.SESSIONS_LIMITED.name());
        bounces = YtDSL.ytIfNull(DSL.sum(tableData.bounces()), BigDecimal.ZERO)
                .as(GdiEntityStats.BOUNCES.name());
        sessionDepth = YtDSL.ytIfNull(DSL.sum(tableData.sessionDepth()), BigDecimal.ZERO)
                .as(GdiEntityStats.SESSION_DEPTH.name());

        ctr =
                YtDSL.ytIf(shows.gt(BigDecimal.ZERO),
                        clicks.mul(PERCENT_DECIMAL_MULT.longValue()).divide(shows),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.CTR.name());
        avgClickCost =
                YtDSL.ytIf(clicks.gt(BigDecimal.ZERO),
                        cost.divide(clicks),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.AVG_CLICK_COST.name());
        avgShowPosition =
                YtDSL.ytIf(firstPageShows.gt(BigDecimal.ZERO),
                        firstPageSumPosShows.mul(DECIMAL_MULT.longValue()).divide(firstPageShows),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.AVG_SHOW_POSITION.name());
        avgClickPosition =
                YtDSL.ytIf(firstPageClicks.gt(BigDecimal.ZERO),
                        firstPageSumPosClicks.mul(DECIMAL_MULT.longValue()).divide(firstPageClicks),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.AVG_CLICK_POSITION.name());
        bounceRate =
                YtDSL.ytIf(sessionNumLimited.gt(BigDecimal.ZERO),
                        bounces.mul(PERCENT_DECIMAL_MULT.longValue()).divide(sessionNumLimited),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.BOUNCE_RATE.name());
        avgDepth =
                YtDSL.ytIf(sessionNumLimited.gt(BigDecimal.ZERO),
                        sessionDepth.mul(DECIMAL_MULT.longValue()).divide(sessionNumLimited),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.AVG_DEPTH.name());
        conversionRate =
                YtDSL.ytIf(clicks.gt(BigDecimal.ZERO),
                        goalsNum.mul(PERCENT_DECIMAL_MULT.longValue()).divide(clicks),
                        DSL.val(BigDecimal.ZERO)).as(GdiEntityStats.CONVERSION_RATE.name());

        /*
        avgGoalCost и profitability могут быть null, чтобы можно было различать отсутствие данных от нулевого значения
         */
        avgGoalCost =
                YtDSL.ytIf(goalsNum.gt(BigDecimal.ZERO),
                        cost.divide(goalsNum),
                        DSL.field("null", BigDecimal.class)).as(GdiEntityStats.AVG_GOAL_COST.name());
        profitability =
                YtDSL.ytIf(cost.gt(BigDecimal.ZERO).and(revenue.gt(BigDecimal.ZERO)),
                        YtDSL.toDouble(revenue.minus(cost)).mul(DSL.val(DECIMAL_MULT.setScale(1, RoundingMode.HALF_UP)))
                                .div(YtDSL.toDouble(cost)),
                        DSL.field("null", BigDecimal.class)).as(GdiEntityStats.PROFITABILITY.name());
        crr =
                YtDSL.ytIf(cost.gt(BigDecimal.ZERO).and(revenue.gt(BigDecimal.ZERO)),
                        YtDSL.toDouble(cost).mul(DSL.val(PERCENT_DECIMAL_MULT.setScale(1, RoundingMode.HALF_UP)))
                                .div(YtDSL.toDouble(revenue)),
                        DSL.field("null", BigDecimal.class)).as(GdiEntityStats.CRR.name());

        this.statOrderFields = generateStatOrderFields();
        this.statFilterProcessor = generateStatFilterProcessor();
    }

    /**
     * Модифицировать переданный контейнер со статистикой таким образом, чтобы вместо всех null, за исключением
     * avgGoalCost и profitability у него были BigDecimal.ZERO
     */
    public static GdiEntityStats addZeros(GdiEntityStats stats) {
        return stats
                .withShows(nvl(stats.getShows(), BigDecimal.ZERO))
                .withClicks(nvl(stats.getClicks(), BigDecimal.ZERO))
                .withCost(nvl(stats.getCost(), BigDecimal.ZERO))
                .withCostWithTax(nvl(stats.getCostWithTax(), BigDecimal.ZERO))
                .withRevenue(nvl(stats.getRevenue(), BigDecimal.ZERO))
                .withGoals(nvl(stats.getGoals(), BigDecimal.ZERO))
                .withCpmPrice(nvl(stats.getCpmPrice(), BigDecimal.ZERO))
                .withFirstPageShows(nvl(stats.getFirstPageShows(), BigDecimal.ZERO))
                .withFirstPageClicks(nvl(stats.getFirstPageClicks(), BigDecimal.ZERO))
                .withFirstPageSumPosShows(nvl(stats.getFirstPageSumPosShows(), BigDecimal.ZERO))
                .withFirstPageSumPosClicks(nvl(stats.getFirstPageSumPosClicks(), BigDecimal.ZERO))
                .withSessions(nvl(stats.getSessions(), BigDecimal.ZERO))
                .withSessionsLimited(nvl(stats.getSessionsLimited(), BigDecimal.ZERO))
                .withSessionDepth(nvl(stats.getSessionDepth(), BigDecimal.ZERO))
                .withBounces(nvl(stats.getBounces(), BigDecimal.ZERO))
                .withCtr(nvl(stats.getCtr(), BigDecimal.ZERO))
                .withAvgClickCost(nvl(stats.getAvgClickCost(), BigDecimal.ZERO))
                .withAvgShowPosition(nvl(stats.getAvgShowPosition(), BigDecimal.ZERO))
                .withAvgClickPosition(nvl(stats.getAvgClickPosition(), BigDecimal.ZERO))
                .withBounceRate(nvl(stats.getBounceRate(), BigDecimal.ZERO))
                .withAvgDepth(nvl(stats.getAvgDepth(), BigDecimal.ZERO))
                .withConversionRate(nvl(stats.getConversionRate(), BigDecimal.ZERO));
    }

    public static List<GdiGoalStats> addZeros(List<GdiGoalStats> stats) {
        return mapList(stats, stat -> stat.withCostPerAction(nvl(stat.getCostPerAction(), BigDecimal.ZERO))
                .withConversionRate(nvl(stat.getConversionRate(), BigDecimal.ZERO))
                .withGoals(nvl(stat.getGoals(), 0L)));
    }

    private JooqFilterProcessor<GdiEntityStatsFilter> generateStatFilterProcessor() {
        return JooqFilterProcessor.<GdiEntityStatsFilter>builder()
                .withFilter(GdiEntityStatsFilter::getMinCost, lessOrEqual(cost, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxCost, greaterOrEqual(cost, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinCostWithTax, lessOrEqual(costWithTax, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxCostWithTax, greaterOrEqual(costWithTax, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinShows, lessOrEqual(shows))
                .withFilter(GdiEntityStatsFilter::getMaxShows, greaterOrEqual(shows))
                .withFilter(GdiEntityStatsFilter::getMinClicks, lessOrEqual(clicks))
                .withFilter(GdiEntityStatsFilter::getMaxClicks, greaterOrEqual(clicks))
                .withFilter(GdiEntityStatsFilter::getMinCtr, lessOrEqual(ctr, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxCtr, greaterOrEqual(ctr, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinAvgClickCost, lessOrEqual(avgClickCost, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxAvgClickCost, greaterOrEqual(avgClickCost, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinAvgShowPosition,
                        lessOrEqual(avgShowPosition, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxAvgShowPosition,
                        greaterOrEqual(avgShowPosition, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinAvgClickPosition,
                        lessOrEqual(avgClickPosition, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxAvgClickPosition,
                        greaterOrEqual(avgClickPosition, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinBounceRate, lessOrEqual(bounceRate, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxBounceRate, greaterOrEqual(bounceRate, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinConversionRate, lessOrEqual(conversionRate, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxConversionRate,
                        greaterOrEqual(conversionRate, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinAvgGoalCost, lessOrEqual(avgGoalCost, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxAvgGoalCost, greaterOrEqual(avgGoalCost, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinGoals, lessOrEqual(goalsNum))
                .withFilter(GdiEntityStatsFilter::getMaxGoals, greaterOrEqual(goalsNum))
                .withFilter(GdiEntityStatsFilter::getMinCpmPrice, lessOrEqual(cpmPrice, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxCpmPrice, greaterOrEqual(cpmPrice, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinAvgDepth, lessOrEqual(avgDepth, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxAvgDepth, greaterOrEqual(avgDepth, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinProfitability, lessOrEqual(profitability, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxProfitability, greaterOrEqual(profitability, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinCrr, lessOrEqual(crr, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxCrr, greaterOrEqual(crr, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMinRevenue, lessOrEqual(revenue, DECIMAL_MULT))
                .withFilter(GdiEntityStatsFilter::getMaxRevenue, greaterOrEqual(revenue, DECIMAL_MULT))
                .build();
    }

    private ImmutableMap<String, Field<?>> generateStatOrderFields() {
        return ImmutableMap.<String, Field<?>>builder()
                .put(GdiStatsOrderBySupportedField.STAT_SHOWS.name(), shows)
                .put(GdiStatsOrderBySupportedField.STAT_CLICKS.name(), clicks)
                .put(GdiStatsOrderBySupportedField.STAT_CTR.name(), ctr)
                .put(GdiStatsOrderBySupportedField.STAT_PROFITABILITY.name(), profitability)
                .put(GdiStatsOrderBySupportedField.STAT_CRR.name(), crr)
                .put(GdiStatsOrderBySupportedField.STAT_COST.name(), cost)
                .put(GdiStatsOrderBySupportedField.STAT_COST_WITH_TAX.name(), costWithTax)
                .put(GdiStatsOrderBySupportedField.STAT_REVENUE.name(), revenue)
                .put(GdiStatsOrderBySupportedField.STAT_GOALS.name(), goalsNum)
                .put(GdiStatsOrderBySupportedField.STAT_CPM_PRICE.name(), cpmPrice)
                .put(GdiStatsOrderBySupportedField.STAT_CONVERSION_RATE.name(), conversionRate)
                .put(GdiStatsOrderBySupportedField.STAT_COST_PER_ACTION.name(),
                        field(GdiEntityStats.COST_PER_ACTION.name())) // расчет только для целей
                .put(GdiStatsOrderBySupportedField.STAT_BOUNCE_RATE.name(), bounceRate)
                .put(GdiStatsOrderBySupportedField.STAT_AVG_CLICK_COST.name(), avgClickCost)
                .put(GdiStatsOrderBySupportedField.STAT_AVG_CLICK_POSITION.name(), avgClickPosition)
                .put(GdiStatsOrderBySupportedField.STAT_AVG_SHOW_POSITION.name(), avgShowPosition)
                .put(GdiStatsOrderBySupportedField.STAT_AVG_DEPTH.name(), avgDepth)
                .put(GdiStatsOrderBySupportedField.STAT_AVG_GOAL_COST.name(), avgGoalCost)
                .build();
    }

    public GridStatTableData<T, T1> getTableData() {
        return tableData;
    }

    //todo возможно стоит удалить этот метод, заменив orderingProcessor на sortingMap в GridRetargetingYtRepository
    // DIRECT-92976

    /**
     * Добавить в билдер условий упорядочивания поля для упорядочивания по статистике
     *
     * @param builder билдер условий
     * @param enumCls класс енама с условиями упорядочивания
     */
    public <E extends Enum<E>> JooqOrderingProcessor.Builder<E> addStatOrdering(
            JooqOrderingProcessor.Builder<E> builder, Class<E> enumCls) {
        for (E e : enumCls.getEnumConstants()) {
            String name = e.name();
            Field<?> orderField = statOrderFields.get(name);

            if (orderField == null) {
                continue;
            }
            builder = builder.withField(e, orderField);
        }
        return builder;
    }

    /**
     * Добавить в мапу условий упорядочивания поля для упорядочивания по статистике
     *
     * @param sortingMap мапа условий упорядочивания
     * @param enumCls    класс енама с условиями упорядочивания
     */
    public <E extends Enum<E>> void addStatOrderingToMap(Map<E, Field<?>> sortingMap, Class<E> enumCls) {
        for (E e : enumCls.getEnumConstants()) {
            String name = e.name();
            Field<?> orderField = statOrderFields.get(name);

            if (orderField == null) {
                continue;
            }
            sortingMap.put(e, orderField);
        }
    }

    /**
     * Получить поля сортировки
     *
     * @param sortingMap     мапа условий упорядочивания
     * @param orderByList    список полей упорядочивания
     * @param defaultOrderBy порядок по умолчанию
     */
    public <E extends Enum<E>> List<OrderField<?>> getOrderFields(Map<E, Field<?>> sortingMap,
                                                                  List<? extends OrderingItemExtended<E>> orderByList,
                                                                  List<OrderField<?>> defaultOrderBy) {
        List<OrderField<?>> orderFields = new ArrayList<>();
        for (OrderingItemExtended<E> orderGroup : orderByList) {
            Field field = sortingMap.get(orderGroup.getField());

            if (orderGroup.getGoalId() != null) {
                field = field(field.getName() + orderGroup.getGoalId());
            }

            orderFields.add(orderGroup.getOrder() == Order.DESC ? field.desc() : field);
        }

        if (orderFields.isEmpty()) {
            orderFields = defaultOrderBy;
        }

        return orderFields;
    }

    /**
     * Получить условие для фильтрации по результирующим статистическим данным
     *
     * @param filter описание условия фильтрации, которое необходимо создать
     */
    @Nullable
    public Condition getStatHavingCondition(@Nullable GdiEntityStatsFilter filter) {
        return statFilterProcessor.apply(filter);
    }

    /**
     * Добавить в условие на группировки фильтрацию статистики по целям
     *
     * @param havingCondition существующие условие на группировки
     * @param goalStats       список фильтров на цели
     */
    public Condition addGoalStatHavingCondition(@Nullable Condition havingCondition,
                                                List<GdiGoalStatsFilter> goalStats) {
        for (GdiGoalStatsFilter goalFilter : goalStats) {
            Field<BigDecimal> conversion = field(getConversionName(goalFilter.getGoalId()), BigDecimal.class);
            Field<BigDecimal> conversionRate = field(getConversionRateName(goalFilter.getGoalId()), BigDecimal.class);
            Field<BigDecimal> costPerAction = field(getCostPerActionName(goalFilter.getGoalId()), BigDecimal.class);

            Condition goalCondition = JooqFilterProcessor.<GdiGoalStatsFilter>builder()
                    .withFilter(GdiGoalStatsFilter::getMinGoals, lessOrEqual(conversion))
                    .withFilter(GdiGoalStatsFilter::getMaxGoals, greaterOrEqual(conversion))
                    .withFilter(GdiGoalStatsFilter::getMinConversionRate, lessOrEqual(conversionRate, DECIMAL_MULT))
                    .withFilter(GdiGoalStatsFilter::getMaxConversionRate, greaterOrEqual(conversionRate, DECIMAL_MULT))
                    .withFilter(GdiGoalStatsFilter::getMinCostPerAction, lessOrEqual(costPerAction, DECIMAL_MULT))
                    .withFilter(GdiGoalStatsFilter::getMaxCostPerAction, greaterOrEqual(costPerAction, DECIMAL_MULT))
                    .build().apply(goalFilter);

            havingCondition = havingCondition == null ? goalCondition : havingCondition.and(goalCondition);
        }
        return havingCondition;
    }

    public Field<BigDecimal> getCost() {
        return cost;
    }

    /**
     * Построить запрос к YT для получения стандартной статистики
     *
     * @param selectFields поля, по которым будет выполняться группировка и суммирование статистики
     * @param whereBase    базовое условие выбора данных для аггрегации (без задания периода)
     * @param startDate    начала периода, за который получаем статистику
     * @param endDate      конец периода, за который получаем статистику (включительно)
     * @param goalIds      список целей
     * @param goalIdsForRevenue список целей, по которым считается доход
     */
    public Select<Record> constructStatSelect(List<Field<?>> selectFields, Condition whereBase,
                                              LocalDate startDate, LocalDate endDate,
                                              Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        List<Field<BigDecimal>> fields = getStatSelectFields();
        return constructStatSelect(selectFields, fields, whereBase, startDate, endDate,
                goalIds, goalIdsForRevenue);
    }

    /**
     * Построить запрос к YT для получения статистики по откруткам
     *
     * @param selectFields поля, по которым будет выполняться группировка и суммирование статистики
     * @param whereBase    базовое условие выбора данных для аггрегации (без задания периода)
     * @param date         день, за который получаем статистику по откруткам
     */
    public Select<Record> constructCostSelect(List<Field<?>> selectFields, Condition whereBase,
                                              LocalDate date) {
        return constructStatSelect(selectFields, Collections.singletonList(cost), whereBase, date, date,
                emptySet(), null);
    }

    /**
     * Построить запрос к YT для получения статистики по откруткам
     *
     * @param selectFields поля, по которым будет выполняться группировка и суммирование статистики
     * @param whereBase    базовое условие выбора данных для аггрегации (без задания периода)
     * @param startDate    начала периода, за который получаем статистику по откруткам
     * @param endDate      конец периода, за который получаем статистику по откруткам (включительно)
     */
    public Select<Record> constructCostSelect(List<Field<?>> selectFields, Condition whereBase,
                                              LocalDate startDate, LocalDate endDate) {
        return constructStatSelect(selectFields, Collections.singletonList(cost), whereBase, startDate, endDate,
                emptySet(), null);
    }

    /**
     * Построить запрос к YT для получения стандартной статистики и целей
     *
     * @param selectFields поля, по которым будет выполняться группировка и суммирование статистики
     * @param whereBase    базовое условие выбора данных для аггрегации (без задания периода)
     * @param startDate    начала периода, за который получаем статистику
     * @param endDate      конец периода, за который получаем статистику (включительно)
     * @param goalIds      список целей
     */
    public SelectHavingStep<Record> constructStatSelectWithPhraseGoals(List<Field<?>> selectFields, Condition whereBase,
                                                                       LocalDate startDate, LocalDate endDate,
                                                                       Set<Long> goalIds,
                                                                       @Nullable Set<Long> goalIdsForRevenue) {
        List<Field<BigDecimal>> fields = getStatSelectFields();
        return constructStatSelectWithPhraseGoals(selectFields, fields, whereBase,
                startDate, endDate, goalIds, goalIdsForRevenue);
    }

    /**
     * Получить список всех вычисляемых статистических полей, которые нужно добавить к запросу
     */
    public List<Field<BigDecimal>> getStatSelectFields() {
        return List.of(shows, clicks, cost, costWithTax, revenue, goalsNum, cpmPrice, firstPageClicks,
                firstPageShows, firstPageSumPosClicks, firstPageSumPosShows, sessionNum, sessionNumLimited,
                bounces, sessionDepth, ctr, avgClickCost, avgShowPosition, avgClickPosition,
                bounceRate, avgDepth, conversionRate, avgGoalCost, profitability, crr);
    }

    /**
     * Создать джойн таблицы со статистическими данными
     *
     * @param table         таблица
     * @param joinCondition условие джойна основной таблицы с таблицей статистики
     * @param startDate     дата начала периода, за который должны вибираться статистические данные
     * @param endDate       дата окончания периода, за который должны вибираться статистические данные (включительно)
     * @param isFlat        true - РСЯ, false - поиск
     */
    public TableOnConditionStep<Record> joinStatColumns(Table<?> table, Condition joinCondition, LocalDate startDate,
                                                        LocalDate endDate, @Nullable Boolean isFlat) {
        Condition flatConditionStep = isFlat == null ? joinCondition : joinCondition
                .and(tableData.isFlat().eq(isFlat ? ytTrue() : ytFalse()));

        return table
                .leftJoin(tableData.table()).on(flatConditionStep.and(periodCondition(startDate, endDate)));
    }

    /**
     * Создать джойн таблиц со статистикой по целям, только для соединения с DirectphrasegoalsstatBs
     *
     * @param joinStep джойн степ
     * @param goalIds  идентификаторы целей
     */
    public TableOnConditionStep<Record> joinGoalStatColumns(TableOnConditionStep<Record> joinStep, Set<Long> goalIds) {
        Directphrasestatv2Bs table = (Directphrasestatv2Bs) tableData.table();
        for (Long goalId : goalIds) {
            DirectphrasegoalsstatBs gs = DIRECTPHRASEGOALSSTAT_BS.as(GOAL_STAT_ALIAS + goalId);
            joinStep = joinStep.leftJoin(gs)
                    .on(getPhraseGoalsCondition(table, gs, goalId));
        }

        logIfGoalsCountExceededThreshold(goalIds.size());

        return joinStep;
    }

    /**
     * Получить поля статистики по целям.
     * Если {@code goalIdsForRevenue} задан, то поле доход
     * будет добавлено только если цель принадлежит {@code goalIdsForRevenue}
     *
     * @param goalIds идентификаторы целей
     * @param goalIdsForRevenue список целей, по которым считается доход
     */
    public List<Field<BigDecimal>> getGoalStatFields(Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        List<Field<BigDecimal>> goalStatFields = new ArrayList<>();
        for (Long goalId : goalIds) {
            GridGoalStatTableData<T1> goalTableData = tableData.goalTableData(goalId);
            //Конверсия
            Field<BigDecimal> goalsNum = YtDSL.ytIfNull(DSL.sum(goalTableData.goalsNum()), ZERO)
                    .as(getConversionName(goalId));

            //Конверсии с показами
            Field<BigDecimal> withShowsGoalsNum = YtDSL.ytIfNull(DSL.sum(goalTableData.withShowsGoalsNum()), ZERO)
                    .as(getConversionWithShowsName(goalId));

            //% конверсии
            Field<BigDecimal> conversionRate =
                    YtDSL.ytIf(clicks.gt(ZERO),
                            goalsNum.mul(PERCENT_DECIMAL_MULT).divide(clicks),
                            DSL.val(ZERO)).as(getConversionRateName(goalId));

            //CPA
            Field<BigDecimal> costPerAction =
                    YtDSL.ytIf(goalsNum.gt(ZERO),
                            cost.divide(goalsNum),
                            DSL.val(ZERO)).as(getCostPerActionName(goalId));

            // Доход по цели
            Field<BigDecimal> revenue;
            if (goalIdsForRevenue == null || goalIdsForRevenue.contains(goalId)) {
                revenue = YtDSL.ytIfNull(DSL.sum(goalTableData.priceCur()), ZERO).as(getRevenueName(goalId));
            } else {
                revenue = DSL.val(ZERO).as(getRevenueName(goalId));
            }
            goalStatFields.addAll(asList(goalsNum, withShowsGoalsNum, conversionRate, costPerAction, revenue));
        }
        return goalStatFields;
    }

    public static Field<Long> getPeriodField(Field<Long> updateTimeField, ReportOptionGroupByDate groupByDate) {
        switch (groupByDate) {
            case WEEK:
                return YtDSL.unixEpochStartOfWeek(updateTimeField, THREE_HOUR);
            case MONTH:
                return YtDSL.unixEpochStartOfMonth(updateTimeField, THREE_HOUR);
            default:
                return updateTimeField;
        }
    }

    public Select<Record> constructConversionsSelect(Collection<Long> campaignIds,
                                                     @Nullable Map<Long, Long> masterBySubId,
                                                     LocalDate startDate,
                                                     LocalDate endDate,
                                                     @Nullable Collection<Long> availableGoalIds,
                                                     @Nullable ReportOptionGroupByDate groupByDate,
                                                     boolean getRevenueOnlyByAvailableGoals,
                                                     boolean getStatsOnlyByAvailableGoals,
                                                     boolean isUacWithOnlyInstallEnabled) {
        var statTable = tableData.goalTableData();

        var campaignId = statTable.effectiveCampaignId(masterBySubId).as(statTable.effectiveCampaignId().getName());
        var updateTime = getPeriodField(statTable.updateTime(),
                nvl(groupByDate, ReportOptionGroupByDate.NONE)).as(statTable.updateTime().getName());
        var goalsNum = statTable.goalsNum();
        var priceCur = statTable.priceCur();
        var goalType = statTable.campaignGoalType();
        var goalId = statTable.goalId();
        var strategyGoalSum = YtDSL.sumIf(goalsNum, goalType.eq(STRATEGY_GOAL_TYPE));
        var meaningfulGoalSum = YtDSL.sumIf(goalsNum, goalType.eq(MEANINGFUL_GOAL_TYPE));
        var filterPriceSum = getRevenueOnlyByAvailableGoals &&
                !getStatsOnlyByAvailableGoals && availableGoalIds != null;
        var strategyPriceSumCondition = goalType.eq(STRATEGY_GOAL_TYPE);
        var meaningfulPriceSumCondition = goalType.eq(MEANINGFUL_GOAL_TYPE);
        if (filterPriceSum) {
            strategyPriceSumCondition = strategyPriceSumCondition.and(goalId.plus(0).in(availableGoalIds));
            meaningfulPriceSumCondition = meaningfulPriceSumCondition.and(goalId.plus(0).in(availableGoalIds));
        }
        var strategyPriceSum = YtDSL.sumIf(priceCur.divide(DECIMAL_MULT), strategyPriceSumCondition);
        var meaningfulPriceSum = YtDSL.sumIf(priceCur.divide(DECIMAL_MULT), meaningfulPriceSumCondition);

        var goalsNumTotal = YtDSL.ytIfNull(
                YtDSL.ytIf(strategyGoalSum.greaterThan(BigDecimal.ZERO), strategyGoalSum, meaningfulGoalSum),
                BigDecimal.ZERO
        ).as(statTable.goalsNum().getName());
        var incomeTotal = YtDSL.ytIfNull(
                YtDSL.ytIf(strategyPriceSum.greaterThan(BigDecimal.ZERO), strategyPriceSum, meaningfulPriceSum),
                BigDecimal.ZERO
        ).as(statTable.priceCur().getName());

        var selectedFields = new ArrayList<>(List.of(campaignId, updateTime, goalsNumTotal, incomeTotal));
        var groupByFields = new ArrayList<>(List.of(campaignId, updateTime));

        if (statTable instanceof DirectGoalPhraseStatData) {
            var groupId = ((DirectGoalPhraseStatData) statTable).groupId()
                    .as(((DirectGoalPhraseStatData) statTable).groupId().getName());
            selectedFields.add(groupId);
            groupByFields.add(groupId);
        }

        var condition = statTable.campaignId()
                .in(GridStatUtils.getAllCampaignIds(campaignIds, masterBySubId))
                .and(YtUtils.periodCondition(updateTime, startDate, endDate))
                .and(goalType.greaterThan(0L));

        if (isUacWithOnlyInstallEnabled) {
            condition = condition.and(goalId.plus(0).eq(INSTALL_CLICK_GOAL_ID));
        } else if (getStatsOnlyByAvailableGoals && availableGoalIds != null) {
            // Добавляем +0, чтобы GoalID использовался для фильтрации, а не как предикат доступа (DIRECT-161954)
            condition = condition.and(goalId.plus(0).in(availableGoalIds));
        }
        return YtDSL.ytContext()
                .select(selectedFields)
                .from(statTable.table())
                .where(condition)
                .groupBy(groupByFields);
    }

    private Condition periodCondition(LocalDate startDate, LocalDate endDate) {
        return YtUtils.periodCondition(tableData.updateTime(), startDate, endDate);
    }


    /**
     * Построить запрос к YT для получения стандартной статистики
     *
     * @param selectFields поля, по которым будет выполняться группировка и суммирование статистики
     * @param statFields   получаемые поля статистики
     * @param whereBase    базовое условие выбора данных для аггрегации (без задания периода)
     * @param startDate    начала периода, за который получаем статистику
     * @param endDate      конец периода, за который получаем статистику (включительно)
     * @param goalIds      список целей
     * @param goalIdsForRevenue список целей, по которым считается доход
     */
    private Select<Record> constructStatSelect(List<Field<?>> selectFields, List<Field<BigDecimal>> statFields,
                                               Condition whereBase, LocalDate startDate, LocalDate endDate,
                                               Set<Long> goalIds, @Nullable Set<Long> goalIdsForRevenue) {
        List<Field<?>> groupBy = constructStatSelectGroupBy(selectFields);

        SelectJoinStep<Record> joinStep =
                constructStatSelectJoinStep(selectFields, statFields, goalIds, goalIdsForRevenue);

        for (Long goalId : goalIds) {
            GridGoalStatTableData goalTableData = tableData.goalTableData(goalId);
            Condition condition = row(tableData.campaignIdHash(), tableData.campaignId(), tableData.updateTime(),
                    tableData.isFlat(), tableData.isMobile(), tableData.currencyId(), tableData.isRmp(),
                    tableData.autobudgetStrategyId(), tableData.dealId(),
                    DSL.val(goalId))
                    .eq(goalTableData.campaignIdHash(), goalTableData.campaignId(),
                            goalTableData.updateTime(),
                            goalTableData.isFlat(), goalTableData.isMobile(), goalTableData.currencyId(),
                            goalTableData.isRmp(), goalTableData.autobudgetStrategyId(),
                            goalTableData.dealId(), goalTableData.goalId());

            joinStep = joinStep.leftJoin(goalTableData.table())
                    .on(condition);
        }

        logIfGoalsCountExceededThreshold(goalIds.size());

        return joinStep
                .where(whereBase
                        .and(periodCondition(startDate, endDate)))
                .groupBy(groupBy);
    }

    private void logIfGoalsCountExceededThreshold(int goalCount) {
        if (goalCount > GOAL_COUNT_THRESHOLD_FOR_LOGGING) {
            logger.info("Select stats from YT with {} joins here: {}", goalCount, getFormattedStackTrace());
        }
    }

    private String getFormattedStackTrace() {
        StringBuilder messageBuilder = new StringBuilder();
        for (StackTraceElement stackTraceElement : Thread.currentThread().getStackTrace()) {
            messageBuilder
                    .append(System.lineSeparator())
                    .append(stackTraceElement.toString());
        }
        return messageBuilder.toString();
    }

    private SelectJoinStep<Record> constructStatSelectJoinStep(List<Field<?>> selectFields,
                                                               List<Field<BigDecimal>> statFields, Set<Long> goalIds,
                                                               @Nullable Set<Long> goalIdsForRevenue) {
        SelectSelectStep<Record> selectStep = YtDSL.ytContext()
                .select(selectFields)
                .select(statFields);
        selectStep.select(getGoalStatFields(goalIds, goalIdsForRevenue));

        return selectStep.from(tableData.table());
    }

    /**
     * Построить запрос к YT для получения стандартной статистики и целей
     *
     * @param selectFields поля, по которым будет выполняться группировка и суммирование статистики
     * @param whereBase    базовое условие выбора данных для аггрегации (без задания периода)
     * @param startDate    начала периода, за который получаем статистику
     * @param endDate      конец периода, за который получаем статистику (включительно)
     * @param goalIds      список целей
     */
    private SelectHavingStep<Record> constructStatSelectWithPhraseGoals(List<Field<?>> selectFields,
                                                                        List<Field<BigDecimal>> statFields,
                                                                        Condition whereBase, LocalDate startDate,
                                                                        LocalDate endDate, Set<Long> goalIds,
                                                                        @Nullable Set<Long> goalIdsForRevenue) {
        List<Field<?>> groupBy = constructStatSelectGroupBy(selectFields);

        SelectJoinStep<Record> joinStep =
                constructStatSelectJoinStep(selectFields, statFields, goalIds, goalIdsForRevenue);

        Directphrasestatv2Bs table = (Directphrasestatv2Bs) tableData.table();
        for (Long goalId : goalIds) {
            DirectphrasegoalsstatBs gs = DIRECTPHRASEGOALSSTAT_BS.as(GOAL_STAT_ALIAS + goalId);
            joinStep = joinStep.leftJoin(gs)
                    .on(getPhraseGoalsCondition(table, gs, goalId));
        }

        logIfGoalsCountExceededThreshold(goalIds.size());

        return joinStep
                .where(whereBase
                        .and(periodCondition(startDate, endDate)))
                .groupBy(groupBy);
    }

    private List<Field<?>> constructStatSelectGroupBy(List<Field<?>> selectFields) {
        return new ArrayList<>(selectFields);
    }

    private Condition getPhraseGoalsCondition(Directphrasestatv2Bs table, DirectphrasegoalsstatBs gs, Long goalId) {
        return row(table.EXPORT_ID, table.GROUP_EXPORT_ID, table.UPDATE_TIME,
                table.PHRASE_EXPORT_ID, table.PHRASE_ID, table.DIRECT_BANNER_ID, table.GOAL_CONTEXT_ID,
                table.IS_FLAT, table.IS_MOBILE, table.CURRENCY_ID, table.IS_RMP,
                ytIfNull(table.AUTOBUDGET_STRATEGY_ID, DEFAULT_AUTOBUDGET_STRATEGY_ID), DSL.val(goalId))
                .eq(gs.EXPORT_ID, gs.GROUP_EXPORT_ID, gs.UPDATE_TIME,
                        gs.PHRASE_EXPORT_ID, gs.PHRASE_ID, gs.DIRECT_BANNER_ID, gs.GOAL_CONTEXT_ID,
                        gs.IS_FLAT, gs.IS_MOBILE, gs.CURRENCY_ID, gs.IS_RMP,
                        ytIfNull(gs.AUTOBUDGET_STRATEGY_ID, DEFAULT_AUTOBUDGET_STRATEGY_ID), gs.GOAL_ID);
    }

    /**
     * Извлечь статистические данные из YT-ноды.
     * Если запрошенных целей ({@code requestedGoalIds}) нет, то доход берется из общей статистики,
     * иначе доход берется как сумма доходов с {@code goalIdsForRevenue} или
     * с {@code requestedGoalIds}, если {@code goalIdsForRevenue} не задан.
     *
     * @param node нода
     */
    public GdiEntityStats extractStatsEntryWithGoalsRevenue(YTreeMapNode node,
                                                            @Nullable Set<Long> requestedGoalIds,
                                                            @Nullable Set<Long> goalIdsForRevenue) {
        GdiEntityStats entity = extractStatsEntry(node);
        if (CollectionUtils.isEmpty(requestedGoalIds)) {
            return entity;
        }
        Set<Long> goalIds = nvl(goalIdsForRevenue, requestedGoalIds);
        BigDecimal goalsRevenue = goalIds.stream()
                .map(goalId -> nvl(extractBsMoney(node, getRevenueName(goalId)), ZERO))
                .reduce(ZERO, BigDecimal::add);

        BigDecimal goalsProfitability = null;
        BigDecimal goalsCrr = null;
        if (nvl(entity.getCost(), ZERO).compareTo(ZERO) > 0 && goalsRevenue.compareTo(ZERO) > 0) {
            goalsProfitability = (goalsRevenue.subtract(entity.getCost()))
                    .multiply(DECIMAL_MULT.setScale(1, RoundingMode.HALF_UP))
                    .divide(entity.getCost(), RoundingMode.HALF_UP);
            goalsProfitability = YtMappingUtils.fromMicros(goalsProfitability.longValue());
            goalsCrr = entity.getCost()
                    .multiply(PERCENT_DECIMAL_MULT.setScale(1, RoundingMode.HALF_UP))
                    .divide(goalsRevenue, RoundingMode.HALF_UP);
            goalsCrr = YtMappingUtils.fromMicros(goalsCrr.longValue());
        }

        return entity.withRevenue(goalsRevenue)
                .withProfitability(goalsProfitability)
                .withCrr(goalsCrr);
    }

    public GdiEntityStats extractStatsEntry(YTreeMapNode node) {
        return new GdiEntityStats()
                .withShows(longNodeBigDecimal(node, shows.getName()))
                .withClicks(longNodeBigDecimal(node, clicks.getName()))
                .withCost(extractBsMoney(node, cost.getName()))
                .withCostMicros(longNodeBigDecimal(node, cost.getName()))
                .withCostWithTax(extractBsMoney(node, costWithTax.getName()))
                .withRevenue(extractBsMoney(node, revenue.getName()))
                .withGoals(longNodeBigDecimal(node, goalsNum.getName()))
                .withCpmPrice(extractBsMoney(node, cpmPrice.getName()))
                .withFirstPageShows(longNodeBigDecimal(node, firstPageShows.getName()))
                .withFirstPageClicks(longNodeBigDecimal(node, firstPageClicks.getName()))
                .withFirstPageSumPosShows(longNodeBigDecimal(node, firstPageSumPosShows.getName()))
                .withFirstPageSumPosClicks(longNodeBigDecimal(node, firstPageSumPosClicks.getName()))
                .withSessions(longNodeBigDecimal(node, sessionNum.getName()))
                .withSessionsLimited(longNodeBigDecimal(node, sessionNumLimited.getName()))
                .withSessionDepth(longNodeBigDecimal(node, sessionDepth.getName()))
                .withBounces(longNodeBigDecimal(node, bounces.getName()))
                .withCtr(extractBsPercent(node, ctr.getName()))
                .withAvgClickCost(extractBsMoney(node, avgClickCost.getName()))
                .withAvgShowPosition(extractBsMoney(node, avgShowPosition.getName()))
                .withAvgClickPosition(extractBsMoney(node, avgClickPosition.getName()))
                .withBounceRate(extractBsPercent(node, bounceRate.getName()))
                .withAvgDepth(extractBsMoney(node, avgDepth.getName()))
                .withConversionRate(extractBsPercent(node, conversionRate.getName()))
                .withAvgGoalCost(extractBsMoney(node, avgGoalCost.getName()))
                .withProfitability(extractBsMoney(node, profitability.getName()))
                .withCrr(extractBsPercent(node, crr.getName()));
    }

    /**
     * Извлечь статистические данные по целям из YT-ноды
     *
     * @param node    нода
     * @param goalIds идентификаторы целей
     */
    public List<GdiGoalStats> extractGoalStatEntries(YTreeMapNode node, Set<Long> goalIds) {
        List<GdiGoalStats> goalStats = new ArrayList<>();
        for (Long goalId : goalIds) {
            long conversion = nvl(longNodeBigDecimal(node, getConversionName(goalId)), ZERO).longValue();
            long conversionWithShow =
                    nvl(longNodeBigDecimal(node, getConversionWithShowsName(goalId)), ZERO).longValue();
            BigDecimal conversionRate = nvl(extractBsPercent(node, getConversionRateName(goalId)), ZERO);
            BigDecimal costPerAction = nvl(extractBsMoney(node, getCostPerActionName(goalId)), ZERO);

            if (conversion != 0 || ZERO.compareTo(costPerAction) != 0) {
                goalStats.add(new GdiGoalStats()
                        .withGoalId(goalId)
                        .withGoals(conversion)
                        .withGoalsWithShows(conversionWithShow)
                        .withConversionRate(conversionRate)
                        .withCostPerAction(costPerAction)
                );
            }
        }
        return goalStats;
    }

    @Nullable
    public GdiGoalConversion extractGoalConversion(YTreeMapNode node) {
        long goalId = node.getOrThrow(DIRECTGRIDGOALSSTAT_BS.GOAL_ID.getName()).longValue();
        long conversion = nvl(node.get(DIRECTGRIDGOALSSTAT_BS.GOALS_NUM.getName()).get().longValue(), 0L);

        if (conversion != 0) {
            return new GdiGoalConversion()
                    .withGoalId(goalId)
                    .withGoals(conversion)
                    .withRevenue(0L);
        }

        return null;
    }

    @Nullable
    public GdiGoalConversion extractGoalConversionWithRevenue(YTreeMapNode node) {
        var goalId = node.getOrThrow(DIRECTGRIDGOALSSTAT_BS.GOAL_ID.getName()).longValue();
        long conversion = nvl(node.get(DIRECTGRIDGOALSSTAT_BS.GOALS_NUM.getName()).get().longValue(), 0L);
        long revenueValue = nvl(node.get(DIRECTGRIDGOALSSTAT_BS.PRICE_CUR.getName()).get().longValue(), 0L);

        if (conversion != 0) {
            return new GdiGoalConversion()
                    .withGoalId(goalId)
                    .withGoals(conversion)
                    .withRevenue(revenueValue);
        }

        return null;
    }

    @Nullable
    public GdiEntityConversion extractEntityConversionWithRevenue(YTreeMapNode node) {
        long conversion = node.get(DIRECTGRIDGOALSSTAT_BS.GOALS_NUM.getName())
                .map(YTreeNode::longValue)
                .orElse(0L);
        long revenueValue = node.get(DIRECTGRIDGOALSSTAT_BS.PRICE_CUR.getName())
                .map(YTreeNode::longValue)
                .orElse(0L);
        LocalDate date = longNodeLocalDate(node, DIRECTGRIDGOALSSTAT_BS.UPDATE_TIME.getName());
        if (conversion == 0 || date == null) {
            return null;
        }

        return new GdiEntityConversion()
                .withDate(date)
                .withGoals(conversion)
                .withRevenue(revenueValue);
    }

    public GdiEntityConversion mergeEntityConversion(GdiEntityConversion first, GdiEntityConversion second) {
        return new GdiEntityConversion()
                .withGoals(first.getGoals() + second.getGoals())
                .withRevenue(first.getRevenue() + second.getRevenue());
    }

    /**
     * Проверяет, содержатся ли в orderByList статистические поля
     */
    public <E extends Enum<E>> boolean hasStatFieldsSort(List<? extends OrderingItem<E>> orderByList) {
        Set<String> orderFields = listToSet(orderByList, x -> x.getField().name());
        return CollectionUtils.containsAny(orderFields, statOrderFields.keySet());
    }

    private String getConversionName(Long goalId) {
        return GdiGoalStats.GOALS.name() + goalId;
    }

    private String getConversionWithShowsName(Long goalId) {
        return GdiGoalStats.GOALS_WITH_SHOWS.name() + goalId;
    }

    private String getConversionRateName(Long goalId) {
        return GdiGoalStats.CONVERSION_RATE.name() + goalId;
    }

    private String getCostPerActionName(Long goalId) {
        return GdiGoalStats.COST_PER_ACTION.name() + goalId;
    }

    private String getRevenueName(Long goalId) {
        return GdiEntityStats.REVENUE.name() + goalId;
    }
}
