package ru.yandex.direct.grid.core.util.stats;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Collection;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.Sets;

import ru.yandex.direct.grid.core.entity.model.GdiEntityConversion;
import ru.yandex.direct.grid.core.entity.model.GdiEntityStats;

import static ru.yandex.direct.utils.CommonUtils.nvl;

public class GridStatUtils {
    private static final BigDecimal PERCENT_MULTIPLIER = BigDecimal.valueOf(100L);

    //Если null, то фолбечимся на дефолтный незначимый идентификатор https://st.yandex-team.ru/DIRECT-163813
    public static final Long DEFAULT_AUTOBUDGET_STRATEGY_ID = -1L;
    public static final Long DEFAULT_DEAL_ID = -1L;

    public static void updateStatWithConversions(GdiEntityStats stats,
                                                 @Nullable GdiEntityConversion entityConversion) {
        var directSumGoals = BigDecimal.ZERO;
        var directSumRevenue = BigDecimal.ZERO;
        if (entityConversion != null) {
            directSumGoals = BigDecimal.valueOf(nvl(entityConversion.getGoals(), 0L));
            directSumRevenue = BigDecimal.valueOf(nvl(entityConversion.getRevenue(), 0L));
        }
        setConversionRelatedFields(stats, directSumGoals, directSumRevenue);
    }

    private static void setConversionRelatedFields(GdiEntityStats stats,
                                                   BigDecimal directSumGoals,
                                                   BigDecimal directSumRevenue) {
        stats.withGoals(directSumGoals);
        stats.withRevenue(directSumRevenue);
        if (!BigDecimal.ZERO.equals(directSumGoals)) {
            stats.withAvgGoalCost(stats.getCost().divide(directSumGoals, RoundingMode.HALF_UP));
        } else {
            stats.withAvgGoalCost(null);
        }
        if (BigDecimal.ZERO.compareTo(directSumRevenue) < 0 && BigDecimal.ZERO.compareTo(stats.getCost()) < 0) {
            stats.setProfitability(
                    directSumRevenue.subtract(stats.getCost()).divide(stats.getCost(), RoundingMode.HALF_UP)
            );
            stats.setCrr(
                    stats.getCost().multiply(PERCENT_MULTIPLIER).divide(directSumRevenue, RoundingMode.HALF_UP)
            );
        } else {
            stats.setCrr(null);
            stats.setProfitability(null);
        }
        if (!BigDecimal.ZERO.equals(stats.getClicks())) {
            stats.setConversionRate(
                    directSumGoals.multiply(PERCENT_MULTIPLIER).divide(stats.getClicks(), 2, RoundingMode.HALF_UP)
            );
        } else {
            stats.withConversionRate(BigDecimal.ZERO);
        }
    }

    /**
     * Собирает коллекцию id для использования в WHERE по campaignId, добавляя к запрошенным кампаниям подлежащие.
     */
    public static Collection<Long> getAllCampaignIds(Collection<Long> campaignIds, @Nullable Map<Long, Long> masterBySubId) {
        if (masterBySubId == null || masterBySubId.isEmpty()) {
            return campaignIds;
        }
        return Sets.union(Set.copyOf(campaignIds), masterBySubId.keySet());
    }

    public static void resetStatWithClicks(@Nullable GdiEntityStats stats) {
        if (stats == null) {
            return;
        }
        stats.withClicks(BigDecimal.ZERO);
        stats.withCtr(BigDecimal.ZERO);
        stats.withAvgClickCost(BigDecimal.ZERO);
        stats.withConversionRate(BigDecimal.ZERO);
    }
}
