package ru.yandex.direct.grid.core.util.yt;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.CompletionException;
import java.util.function.Consumer;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.proxy.request.ColumnFilter;
import ru.yandex.yt.ytclient.proxy.request.GetNode;

import static com.google.common.base.Preconditions.checkState;
import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.ATTR_MIN_TIMESTAMPS;
import static ru.yandex.direct.grid.core.entity.recommendation.RecommendationTablesUtils.deserializeMinTimestamps;

/**
 * Таска для {@link java.util.Timer}, которая при запуске
 * определяет для каждого кластера информацию о его свежести и передает её потребителю
 */
@ParametersAreNonnullByDefault
public class YtClusterFreshnessLoader {
    private static final Logger logger = LoggerFactory.getLogger(YtClusterFreshnessLoader.class);

    private final YtProvider ytProvider;
    private final DirectYtDynamicConfig dynamicConfig;
    private final String recommendationsTablePath;
    private final YtClusterFreshnessRepository ytClusterFreshnessRepository;

    private Map<YtCluster, YtDynamicOperator> clusterToOperator = new HashMap<>();

    private final Timer timer;

    private volatile List<Consumer<Map<YtCluster, ClusterFreshnessInfo>>> subscribers = new ArrayList<>();

    public YtClusterFreshnessLoader(YtProvider ytProvider, DirectYtDynamicConfig dynamicConfig,
                                    YtClusterFreshnessRepository ytClusterFreshnessRepository) {
        this.dynamicConfig = dynamicConfig;
        this.ytProvider = ytProvider;
        this.recommendationsTablePath = dynamicConfig.tables().recommendations().recommendationsTablePath();
        this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;

        this.timer = new Timer(true);
        long period = dynamicConfig.getClusterRefreshPeriod().toMillis();
        timer.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                runUpdateFreshness();
            }
        }, period, period);
    }

    /**
     * Пытаемся проинициализировать операторы для нужных кластеров.
     * В случае успеха запоминаем созданный {@link YtDynamicOperator} в {@link #clusterToOperator}
     */
    void fillClusterToOperator() {
        for (YtCluster cluster : dynamicConfig.getClusters()) {
            // к clusterToOperator обращаемся из одного потока, можем позволить себе модифицировать его inplace
            clusterToOperator.computeIfAbsent(cluster, this::getDynamicOperatorSafely);
        }
    }

    /**
     * Exception-safe обёртка над {@link YtProvider#getDynamicOperator(ru.yandex.direct.ytwrapper.model.YtCluster)}.
     * <p>
     * Если не удалось создать {@code YtDynamicOperator}, верёнтся {@code null}
     */
    @Nullable
    private YtDynamicOperator getDynamicOperatorSafely(YtCluster cluster) {
        try {
            return ytProvider.getDynamicOperator(cluster);
        } catch (RuntimeException e) {
            logger.error("Can't init operator for cluster {}", cluster, e);
            return null;
        }
    }

    void runUpdateFreshness() {
        try {
            Map<YtCluster, ClusterFreshnessInfo> freshnessMap = getAllClustersFreshness();

            // Если по какой-то причине мы получили пустой набор данных про кластера,
            // мы не можем его передать consumer'у, т.к. это приведёт к недоступности сервиса
            // В этом случае лучше подождать следующего обновления
            checkState(!freshnessMap.isEmpty());

            List<Consumer<Map<YtCluster, ClusterFreshnessInfo>>> subscribers = this.subscribers;
            subscribers.forEach(subscriber -> subscriber.accept(freshnessMap));
        } catch (Exception e) {
            // Нужно обязательно поймать исключение, т.к. иначе таймер больше не сработает
            logger.error("Unhandled exception in timer task", e);
        }
    }

    /**
     * Определяет, до всех ли кластеров смогли достучаться
     */
    private boolean allOperatorInitialized() {
        return clusterToOperator.size() == dynamicConfig.getClusters().size();
    }

    /**
     * Получить информацию о свежести для всех имеющихся кластеров.
     * Если по какой-то причине мы не смогли получить такую информацию ни для одного из кластеров,
     * будет выброшено исключение.
     */
    Map<YtCluster, ClusterFreshnessInfo> getAllClustersFreshness() {
        if (!allOperatorInitialized()) {
            fillClusterToOperator();
        }
        Map<YtCluster, ClusterFreshnessInfo> freshnessMap = new HashMap<>();
        for (YtCluster ytCluster : clusterToOperator.keySet()) {
            ClusterFreshnessInfo freshness = loadClusterFreshness(ytCluster);
            if (freshness != null) {
                freshnessMap.put(ytCluster, freshness);
            }
        }

        if (freshnessMap.isEmpty()) {
            throw new YtClusterSelectionException("No available clusters");
        }

        return freshnessMap;
    }

    /**
     * Загружает информацию о свежести для выбранного кластера
     * Возвращает null, если данные от кластера получить не удалось
     */
    @Nullable
    private ClusterFreshnessInfo loadClusterFreshness(YtCluster cluster) {
        YtDynamicOperator ytOperator = clusterToOperator.get(cluster);

        // NB: это можно улучшить, запустив обе операции загрузки одновременно и затем ожидая их в allOf()
        @Nullable Map<Integer, Long> shardToTimestamp = ytClusterFreshnessRepository.loadShardToTimestamp(ytOperator);

        // Если не удалось получить эту информацию, то и далее можно уже не идти
        if (shardToTimestamp == null) {
            return null;
        }

        long recommendationsTimestamp = loadRecommendationsTimestamp(ytOperator);

        return new ClusterFreshnessInfo(shardToTimestamp, recommendationsTimestamp);
    }

    /**
     * Возвращает timestamp (в ms) свежести основной таблицы рекомендаций
     */
    private long loadRecommendationsTimestamp(YtDynamicOperator ytOperator) {
        try {
            YTreeNode node = ytOperator.getYtClient()
                    .getNode(new GetNode(recommendationsTablePath).setAttributes(
                            ColumnFilter.of(ATTR_MIN_TIMESTAMPS))
                    ).join(); // IGNORE-BAD-JOIN DIRECT-149116
            Map<Integer, Long> minTimestamps = deserializeMinTimestamps(node.getAttributeOrThrow(ATTR_MIN_TIMESTAMPS));
            Optional<Long> maxTs = minTimestamps.values().stream().max(Long::compare);

            // Если в метаданных таблицы не оказалось таймстемпов, вернём 0
            // В таблице таймстемпы хранятся как unix time в секундах, но для сравнения с таймстемпами
            // синхронизатора их надо перевести в миллисекунды
            return 1000 * maxTs.orElse(0L);
        } catch (CompletionException | NoSuchElementException e) {
            logger.error("Error when loading recommendations metadata", e);

            // Если произошла ошибка (например, таблицы не существует или в ней нет нужного атрибута), вернём 0
            return 0L;
        }
    }

    @SuppressWarnings("WeakerAccess")
    public synchronized void addHandler(Consumer<Map<YtCluster, ClusterFreshnessInfo>> subscriber) {
        List<Consumer<Map<YtCluster, ClusterFreshnessInfo>>> newList = new ArrayList<>(this.subscribers);
        checkState(!newList.contains(subscriber));
        newList.add(subscriber);
        this.subscribers = newList;
    }

    @SuppressWarnings("WeakerAccess")
    public synchronized void removeHandler(Consumer<Map<YtCluster, ClusterFreshnessInfo>> subscriber) {
        List<Consumer<Map<YtCluster, ClusterFreshnessInfo>>> newList = new ArrayList<>(this.subscribers);
        newList.remove(subscriber);
        this.subscribers = newList;
    }

    @PreDestroy
    public void stopTimer() {
        timer.cancel();
    }
}
