package ru.yandex.direct.grid.core.util.yt;

import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import one.util.streamex.EntryStream;
import org.jooq.Select;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.YtDynamicConfig;
import ru.yandex.direct.ytwrapper.dynamic.context.YtDynamicContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.selector.ClusterChooser;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;

@ParametersAreNonnullByDefault
public class YtDynamicSupport {
    private static final int SHARD_WHEN_NOT_PROVIDED = 1;

    private static final Logger logger = LoggerFactory.getLogger(YtDynamicSupport.class);
    public static final int FRESHNESS_THRESHOLD_MINUTES = 60;
    public static final int MAX_CLUSTERS_PER_SHARD = 2;

    private final ShardHelper shardHelper;
    private final YtDynamicContextProvider<ShardParam> gridYtDynamicContextProvider;
    private final YtCluster firstClusterFromConfig;
    private final YtClusterFreshnessLoader freshnessLoader;
    private final Consumer<Map<YtCluster, ClusterFreshnessInfo>> handler;

    private volatile Map<Integer, List<YtCluster>> shardToClusters;

    public YtDynamicSupport(ShardHelper shardHelper, YtProvider ytProvider, DirectYtDynamicConfig dynamicConfig,
                            YtClusterFreshnessLoader freshnessLoader,
                            YtDynamicConfig ytDynamicConfig) {
        this.shardHelper = shardHelper;

        checkArgument(!dynamicConfig.getClusters().isEmpty(), "We need at least one cluster");
        this.firstClusterFromConfig = dynamicConfig.getClusters().iterator().next();

        // Конфигурируем DynamicContextProvider
        GridClusterChooser gridClusterChooser = new GridClusterChooser();
        gridYtDynamicContextProvider = new YtDynamicContextProvider<>(gridClusterChooser, ytProvider,
                ytDynamicConfig.defaultSelectRowsTimeout());

        // Подписываемся на события от freshnessLoader
        this.freshnessLoader = freshnessLoader;
        this.handler = this::setFreshnessMap;
        freshnessLoader.addHandler(this.handler);
    }

    private class GridClusterChooser implements ClusterChooser<ShardParam> {
        @Nonnull
        @Override
        public Optional<YtCluster> getCluster(@Nullable ShardParam param) {
            if (param == null) {
                return Optional.ofNullable(getShardToClusters().get(SHARD_WHEN_NOT_PROVIDED))
                        .map(clusters -> clusters.get(0));
            }
            int shard = param.shard();
            return Optional.ofNullable(getShardToClusters().get(shard))
                    .map(clusters -> clusters.get(0));
        }

        @Nonnull
        @Override
        public Optional<List<YtCluster>> getClustersOrdered(@Nullable ShardParam param) {
            if (param == null) {
                return Optional.ofNullable(getShardToClusters().get(SHARD_WHEN_NOT_PROVIDED));
            }
            int shard = param.shard();
            return Optional.ofNullable(getShardToClusters().get(shard));
        }
    }

    private static class ShardParam {
        private final int shard;

        private ShardParam(int shard) {
            this.shard = shard;
        }

        static ShardParam shard(int shard) {
            return new ShardParam(shard);
        }

        public int shard() {
            return shard;
        }
    }

    public UnversionedRowset selectRows(Select query) {
        return selectRows(SHARD_WHEN_NOT_PROVIDED, query);
    }

    public UnversionedRowset selectRows(int shard, Select query) {
        return selectRows(shard, query, false);
    }

    public UnversionedRowset selectRows(int shard, Select query, boolean withTotalStats) {
        return gridYtDynamicContextProvider.getContext(ShardParam.shard(shard)).executeSelect(query, withTotalStats);
    }

    @Nonnull
    private Map<Integer, List<YtCluster>> getShardToClusters() {
        if (shardToClusters == null) {
            // Считаем что для старта нам достаточно первого попавшегося кластера
            // Если кластеров больше, чем один, то значение будет регулярно обновляться через YtClusterFreshnessLoader
            //noinspection ConstantConditions
            Map<Integer, List<YtCluster>> clusterMap = shardHelper.dbShards().stream()
                    .collect(toMap(Function.identity(), s -> Collections.singletonList(firstClusterFromConfig)));
            setShardToClusters(clusterMap);
        }
        return shardToClusters;
    }

    private void setShardToClusters(Map<Integer, List<YtCluster>> shardToClusters) {
        logger.debug("Setting shard operator to {}", shardToClusters);
        this.shardToClusters = shardToClusters;
    }

    private void setFreshnessMap(Map<YtCluster, ClusterFreshnessInfo> freshnessMap) {
        // Сюда не должно приезжать пустое соответствие
        checkArgument(!freshnessMap.isEmpty());

        // Для каждого шарда получаем самый свежий кластер (для обычных запросов)
        setShardToClusters(calculateShardToClusters(freshnessMap));
    }

    static Map<Integer, List<YtCluster>> calculateShardToClusters(Map<YtCluster, ClusterFreshnessInfo> freshnessMap) {
        checkArgument(!freshnessMap.isEmpty());
        long currentTimestampSeconds = System.currentTimeMillis() / 1000;
        long clusterTimestampLowerLimit = currentTimestampSeconds - FRESHNESS_THRESHOLD_MINUTES * 60;

        return EntryStream.of(freshnessMap)
                // Формируем список соответствий shard -> {YtCluster, shardTs}
                .flatMap(entry -> EntryStream.of(entry.getValue().shardTimestamps)
                        .mapValues(v -> new YtClusterTs(entry.getKey(), v))
                )
                // группируем по номеру шарда инфу о свежести
                .groupingBy(Map.Entry::getKey, mapping(Map.Entry::getValue, toList()))
                .entrySet().stream()
                .collect(toMap(Map.Entry::getKey, e -> e.getValue().stream()
                        // Для каждого shard сортируем список кластеров по убыванию timestamp
                        .sorted(Comparator.comparingLong(ts -> ((YtClusterTs) ts).ts).reversed())
                        // отсечь кластеры старше 10 минут, и брать только 2
                        .filter(ct -> ct.ts >= clusterTimestampLowerLimit)
                        .limit(MAX_CLUSTERS_PER_SHARD)
                        // Осталось из значений убрать timestamp (нам нужен только YtCluster)
                        .map(ct -> ct.ytCluster)
                        .collect(toList())));
    }

    private static class YtClusterTs {
        final YtCluster ytCluster;
        final long ts;

        YtClusterTs(YtCluster ytCluster, long ts) {
            this.ytCluster = ytCluster;
            this.ts = ts;
        }
    }

    @PreDestroy
    public void unsubscribe() {
        freshnessLoader.removeHandler(this.handler);
    }
}
