package ru.yandex.direct.grid.core.util.yt;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;

import javax.annotation.Nonnull;
import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import com.google.common.base.Preconditions;
import one.util.streamex.EntryStream;
import org.apache.commons.lang3.tuple.Pair;
import org.jooq.Select;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.dbutil.sharding.ShardHelper;
import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.dynamic.YtDynamicConfig;
import ru.yandex.direct.ytwrapper.dynamic.context.YtDynamicContext;
import ru.yandex.direct.ytwrapper.dynamic.context.YtDynamicContextProvider;
import ru.yandex.direct.ytwrapper.dynamic.selector.ClusterChooser;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;

import static com.google.common.base.Preconditions.checkArgument;
import static java.lang.Long.min;
import static java.util.Collections.singletonList;
import static java.util.Comparator.comparingLong;
import static java.util.stream.Collectors.collectingAndThen;
import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.mapping;
import static java.util.stream.Collectors.toList;
import static java.util.stream.Collectors.toMap;
import static java.util.stream.Collectors.toSet;

@ParametersAreNonnullByDefault
public class YtRecommendationsDynamicSupport {
    private static final Logger logger = LoggerFactory.getLogger(YtRecommendationsDynamicSupport.class);

    public static final int MAX_CLUSTERS_PER_SHARD = 2;

    private final ShardHelper shardHelper;
    private final YtDynamicContextProvider<ShardParam> gridYtDynamicContextProvider;
    private final YtCluster firstClusterFromConfig;
    private final YtClusterFreshnessLoader freshnessLoader;
    private final Consumer<Map<YtCluster, ClusterFreshnessInfo>> handler;

    private volatile Map<Integer, List<YtCluster>> recommendationsShardToCluster;

    public YtRecommendationsDynamicSupport(ShardHelper shardHelper, YtProvider ytProvider,
                                           DirectYtDynamicConfig dynamicConfig,
                                           YtClusterFreshnessLoader freshnessLoader,
                                           YtDynamicConfig ytDynamicConfig) {
        this.shardHelper = shardHelper;

        checkArgument(!dynamicConfig.getClusters().isEmpty(), "We need at least one cluster");
        this.firstClusterFromConfig = dynamicConfig.getClusters().iterator().next();

        // Конфигурируем DynamicContextProvider
        GridClusterChooser gridClusterChooser = new GridClusterChooser();
        gridYtDynamicContextProvider = new YtDynamicContextProvider<>(gridClusterChooser, ytProvider,
                ytDynamicConfig.defaultSelectRowsTimeout());

        // Подписываемся на события от freshnessLoader
        this.freshnessLoader = freshnessLoader;
        this.handler = this::setFreshnessMap;
        freshnessLoader.addHandler(this.handler);
    }

    private class GridClusterChooser implements ClusterChooser<ShardParam> {
        @Nonnull
        @Override
        public Optional<YtCluster> getCluster(@Nullable ShardParam param) {
            if (param == null) {
                return Optional.empty();
            }
            int shard = param.shard();
            return Optional.ofNullable(getRecommendationsShardToCluster().get(shard))
                    .map(clusters -> clusters.get(0));
        }

        @Nonnull
        @Override
        public Optional<List<YtCluster>> getClustersOrdered(@Nullable ShardParam param) {
            if (param == null) {
                return Optional.empty();
            }
            int shard = param.shard();
            return Optional.ofNullable(getRecommendationsShardToCluster().get(shard));
        }
    }

    private static class ShardParam {
        private final int shard;

        private ShardParam(int shard) {
            this.shard = shard;
        }

        static ShardParam forShard(int shard) {
            return new ShardParam(shard);
        }

        public int shard() {
            return shard;
        }
    }

    /**
     * Метод для загрузки данных по рекомендациям.
     * Учитывает свежесть рекомендаций при выборе кластера (помимо свежести шардов)
     */
    public UnversionedRowset selectRows(int shard, Select query) {
        return gridYtDynamicContextProvider.getContext(ShardParam.forShard(shard)).executeSelect(query);
    }

    /**
     * Метод для массовой загрузки рекомендаций по набору шардов.
     * Шарды группируются по самым свежим для них кластерам и далее на каждый такой кластер отправляется по одному запросу.
     * Результаты объединяются и возвращаются.
     * Так как заранее группировка шардов по свежайшим кластерам неизвестна, запрос приходится предоставлять посредством провайдера.
     *
     * @param allShards              Все шарды, участвующие в запросе
     * @param queryForShardsProvider Провайдер, который будет собирать запросы для наборов шардов
     */
    public UnversionedRowset selectRows(Set<Integer> allShards, Function<Set<Integer>, Select> queryForShardsProvider) {
        Preconditions.checkArgument(!allShards.isEmpty());

        // Группируем шарды по обслуживающим их кластерам
        Map<YtDynamicContext, Set<Integer>> shardsByContext = allShards.stream()
                .map(shard -> Pair.of(shard, gridYtDynamicContextProvider.getContext(ShardParam.forShard(shard))))
                .collect(groupingBy(Pair::getRight, collectingAndThen(
                        toList(), list -> list.stream().map(Pair::getLeft).collect(toSet())
                )));

        // NB: можно сделать параллельное выполнение
        List<UnversionedRowset> rowSets = new ArrayList<>();
        shardsByContext.forEach((ytDynamicContext, shards) -> {
            Select query = queryForShardsProvider.apply(shards);
            rowSets.add(ytDynamicContext.executeSelect(query));
        });

        // Объединяем в один результат
        return new UnversionedRowset(
                rowSets.get(0).getSchema(),
                rowSets.stream().map(UnversionedRowset::getRows).flatMap(Collection::stream).collect(toList()));
    }

    private Map<Integer, List<YtCluster>> getRecommendationsShardToCluster() {
        if (recommendationsShardToCluster == null) {
            Map<Integer, List<YtCluster>> clusterMap = shardHelper.dbShards().stream()
                    .collect(toMap(Function.identity(), s -> singletonList(firstClusterFromConfig)));
            setRecommendationsShardToCluster(clusterMap);
        }
        return recommendationsShardToCluster;
    }

    private void setRecommendationsShardToCluster(Map<Integer, List<YtCluster>> recommendationsShardToCluster) {
        logger.debug("Settings recommendations shard->cluster to {}", recommendationsShardToCluster);
        this.recommendationsShardToCluster = recommendationsShardToCluster;
    }

    private void setFreshnessMap(Map<YtCluster, ClusterFreshnessInfo> freshnessMap) {
        // Сюда не должно приезжать пустое соответствие
        checkArgument(!freshnessMap.isEmpty());

        // Для каждого шарда получаем самый свежий кластер (для запросов к таблицам рекомендаций)
        Map<Integer, List<YtCluster>> recommendationsShardToCluster = calculateRecommendationsShardToCluster(freshnessMap);
        setRecommendationsShardToCluster(recommendationsShardToCluster);

    }

    static Map<Integer, List<YtCluster>> calculateRecommendationsShardToCluster(
            Map<YtCluster, ClusterFreshnessInfo> freshnessMap) {
        checkArgument(!freshnessMap.isEmpty());

        return EntryStream.of(freshnessMap)
                // Делаем список соответствий shard -> {YtCluster, shardTs, recommendationTs}
                .flatMap(entry -> EntryStream.of(entry.getValue().shardTimestamps)
                        .mapValues(v -> new YtClusterTs(entry.getKey(), v, entry.getValue().recommendationsTimestamp))
                )
                .groupingBy(Map.Entry::getKey, mapping(Map.Entry::getValue, toList()))
                .entrySet().stream()
                .collect(toMap(Map.Entry::getKey, e -> e.getValue().stream()
                        // Для каждого shard сортируем список кластеров по убыванию Long.min(shardTs, recommendationTs)
                        .sorted(comparingLong(ts -> min(((YtClusterTs) ts).shardTs, ((YtClusterTs) ts).recommendationsTs)).reversed())
                        .limit(MAX_CLUSTERS_PER_SHARD)
                        // Осталось из значений убрать timestamp (нам нужен только YtCluster)
                        .map(ct -> ct.ytCluster)
                        .collect(toList())));
    }

    private static class YtClusterTs {
        final YtCluster ytCluster;
        final long shardTs;
        final long recommendationsTs;

        YtClusterTs(YtCluster ytCluster, long shardTs, long recommendationsTs) {
            this.ytCluster = ytCluster;
            this.shardTs = shardTs;
            this.recommendationsTs = recommendationsTs;
        }
    }

    @PreDestroy
    public void unsubscribe() {
        freshnessLoader.removeHandler(this.handler);
    }

}
