package ru.yandex.direct.grid.core.util.yt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import one.util.streamex.StreamEx;

import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.model.MysqlSyncStates;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.model.YtCluster;

import static java.util.Collections.emptyList;
import static org.apache.commons.collections4.CollectionUtils.isNotEmpty;
import static ru.yandex.direct.grid.core.entity.sync.repository.YtSyncStateRepository.shardToDbName;

@ParametersAreNonnullByDefault
public class YtSyncStatesDynamicSupport {

    private final YtSyncStatesLoader ytSyncStatesLoader;
    private final DirectYtDynamicConfig dynamicConfig;

    public YtSyncStatesDynamicSupport(YtSyncStatesLoader ytSyncStatesLoader,
                                      DirectYtDynamicConfig dynamicConfig) {
        this.ytSyncStatesLoader = ytSyncStatesLoader;
        this.dynamicConfig = dynamicConfig;
    }

    public Optional<String> getGtidSetForShard(int shard) {
        Map<YtCluster, List<MysqlSyncStates>> clusterToSyncStates = getPrecalculatedSyncStatesForAllClusters();

        if (clusterToSyncStates.isEmpty()) {
            return Optional.empty();
        }
        Map<Integer, List<YtCluster>> shardToClusters = calculateShardToClusters(clusterToSyncStates);

        return calculateGtidSetForShard(shard, shardToClusters, clusterToSyncStates);
    }

    private Map<YtCluster, List<MysqlSyncStates>> getPrecalculatedSyncStatesForAllClusters() {
        Map<YtCluster, List<MysqlSyncStates>> clusterToSyncStates = new HashMap<>();

        for (YtCluster ytCluster : dynamicConfig.getClusters()) {
            List<MysqlSyncStates> syncStates = ytSyncStatesLoader.getSyncStatesForCluster(ytCluster);
            if (isNotEmpty(syncStates)) {
                clusterToSyncStates.put(ytCluster, syncStates);
            }
        }
        return clusterToSyncStates;
    }

    private static Map<Integer, List<YtCluster>> calculateShardToClusters(
            Map<YtCluster, List<MysqlSyncStates>> clusterToSyncStates) {

        Map<YtCluster, ClusterFreshnessInfo> freshnessMap = EntryStream.of(clusterToSyncStates)
                .mapValues(YtSyncStatesDynamicSupport::convertToClusterFreshness)
                .toMap();
        return YtDynamicSupport.calculateShardToClusters(freshnessMap);
    }

    private static ClusterFreshnessInfo convertToClusterFreshness(List<MysqlSyncStates> syncStates) {
        Map<Integer, Long> shardToTimestamp = StreamEx.of(syncStates)
                .mapToEntry(MysqlSyncStates::getDbName, MysqlSyncStates::getLastTimestamp)
                .mapKeys(YtClusterFreshnessRepository::dbNameToShard)
                .toMap();
        return new ClusterFreshnessInfo(shardToTimestamp, 0L);
    }

    private static Optional<String> calculateGtidSetForShard(
            int shard, Map<Integer, List<YtCluster>> shardToClusters,
            Map<YtCluster, List<MysqlSyncStates>> clusterToSyncStates) {

        String dbNameForShard = shardToDbName(shard);
        List<YtCluster> clustersByPriority = shardToClusters.getOrDefault(shard, emptyList());

        for (YtCluster ytCluster : clustersByPriority) {
            List<MysqlSyncStates> syncStates = clusterToSyncStates.getOrDefault(ytCluster, emptyList());

            for (MysqlSyncStates syncState : syncStates) {
                if (dbNameForShard.equals(syncState.getDbName())) {
                    return Optional.of(syncState.getGtidSet());
                }
            }
        }
        return Optional.empty();
    }
}
