package ru.yandex.direct.grid.core.util.yt;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;
import javax.annotation.PreDestroy;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.ytcomponents.config.DirectYtDynamicConfig;
import ru.yandex.direct.ytcomponents.model.MysqlSyncStates;
import ru.yandex.direct.ytcomponents.repository.YtClusterFreshnessRepository;
import ru.yandex.direct.ytwrapper.client.YtProvider;
import ru.yandex.direct.ytwrapper.model.YtCluster;
import ru.yandex.direct.ytwrapper.model.YtDynamicOperator;

/**
 * Таска для {@link Timer}, которая при запуске
 * определяет для каждого кластера информацию о его свежести и передает её потребителю
 */
@ParametersAreNonnullByDefault
public class YtSyncStatesLoader {
    private static final Logger logger = LoggerFactory.getLogger(YtSyncStatesLoader.class);

    private final YtProvider ytProvider;
    private final DirectYtDynamicConfig dynamicConfig;
    private final YtClusterFreshnessRepository ytClusterFreshnessRepository;

    private final Map<YtCluster, YtDynamicOperator> clusterToOperator = new HashMap<>();
    private final ConcurrentMap<YtCluster, List<MysqlSyncStates>> clusterToSyncStatesCache = new ConcurrentHashMap<>();

    private final Timer timer;

    public YtSyncStatesLoader(YtProvider ytProvider,
                              DirectYtDynamicConfig dynamicConfig,
                              YtClusterFreshnessRepository ytClusterFreshnessRepository) {
        this.ytProvider = ytProvider;
        this.dynamicConfig = dynamicConfig;
        this.ytClusterFreshnessRepository = ytClusterFreshnessRepository;

        // TODO: вместо Timer использовать ScheduledExecutorService
        // TODO: period 20s -> 1s
        this.timer = new Timer(true);
        long period = dynamicConfig.getClusterRefreshPeriod().toMillis();
        timer.scheduleAtFixedRate(new TimerTask() {
            @Override
            public void run() {
                runUpdateSyncStates();
            }
        }, period, period);
    }

    /**
     * Пытаемся проинициализировать операторы для нужных кластеров.
     * В случае успеха запоминаем созданный {@link YtDynamicOperator} в {@link #clusterToOperator}
     */
    void fillClusterToOperator() {
        for (YtCluster cluster : dynamicConfig.getClusters()) {
            // к clusterToOperator обращаемся из одного потока, можем позволить себе модифицировать его inplace
            clusterToOperator.computeIfAbsent(cluster, this::getDynamicOperatorSafely);
        }
    }

    /**
     * Exception-safe обёртка над {@link YtProvider#getDynamicOperator(YtCluster)}.
     * <p>
     * Если не удалось создать {@code YtDynamicOperator}, верёнтся {@code null}
     */
    @Nullable
    private YtDynamicOperator getDynamicOperatorSafely(YtCluster cluster) {
        try {
            return ytProvider.getDynamicOperator(cluster);
        } catch (RuntimeException e) {
            logger.error("Can't init operator for cluster {}", cluster, e);
            return null;
        }
    }

    void runUpdateSyncStates() {
        try {
            updateSyncStates();

        } catch (RuntimeException e) {
            // Нужно обязательно поймать исключение, т.к. иначе таймер больше не сработает
            logger.error("Unhandled exception in timer task", e);
        }
    }

    /**
     * Определяет, до всех ли кластеров смогли достучаться
     */
    private boolean allOperatorInitialized() {
        return clusterToOperator.size() == dynamicConfig.getClusters().size();
    }

    private void updateSyncStates() {
        if (!allOperatorInitialized()) {
            fillClusterToOperator();
        }

        for (YtCluster ytCluster : clusterToOperator.keySet()) {
            YtDynamicOperator ytOperator = clusterToOperator.get(ytCluster);

            Optional<List<MysqlSyncStates>> optionalSyncStates =
                    ytClusterFreshnessRepository.loadSyncStatesForAllShards(ytOperator);

            optionalSyncStates.ifPresent(syncStates -> clusterToSyncStatesCache.put(ytCluster, syncStates));
        }
    }

    @Nullable
    List<MysqlSyncStates> getSyncStatesForCluster(YtCluster cluster) {
        return clusterToSyncStatesCache.get(cluster);
    }

    @PreDestroy
    public void stopTimer() {
        timer.cancel();
    }
}
