package ru.yandex.direct.grid.core.util.yt.mapping;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.jooq.Field;
import org.jooq.Table;
import org.jooq.TableRecord;
import org.jooq.impl.CustomRecord;
import org.jooq.types.ULong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jooqmapper.JooqMapperWithSupplier;
import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.inside.yt.kosher.impl.ytree.builder.YTree;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;
import ru.yandex.yt.ytclient.tables.ColumnValueType;
import ru.yandex.yt.ytclient.tables.TableSchema;
import ru.yandex.yt.ytclient.wire.UnversionedRow;
import ru.yandex.yt.ytclient.wire.UnversionedRowset;
import ru.yandex.yt.ytclient.wire.UnversionedValue;

/**
 * Надстройка над {@link JooqMapperWithSupplier}, которая позволяет конвертировать
 * ноды, полученные из YT после выполнения запросов к динамическим таблицам, во внутренние модели приложения.
 * На текущий момент поддерживает конвертацию только для одной таблицы
 *
 * @param <M> тип внутренней модели
 * @param <R> тип jooq-записи таблицы, описывающей YT-таблиц
 *
 * @deprecated Используйте {@link YtReader}.
 * Этот класс не поддерживает чтение больше, чем из одной таблицы, и поддержку этого в полной мере нельзя добавить с
 * полным сохранением обратной совместимости - например, для этого не подходит текущий механизм формирования алиасов:
 * т.к. они не включают в себя имя таблицы, при чтении полей с одинаковым названием из разных таблиц все ломается -
 * так что нужно видоизменить генерируемые запросы. {@link YtReader} более универсальный и поддерживает больше кейсов.
 */
@ParametersAreNonnullByDefault
@Deprecated
public class YtFieldMapper<M extends Model, R extends TableRecord<R>> {
    private static final Logger logger = LoggerFactory.getLogger(YtFieldMapper.class);

    private final JooqReaderWithSupplier<M> jooqReader;
    private final Table<R> table;

    /**
     * @param jooqReader настроенный JooqReader
     * @param table      объект таблицы, для которой работает маппер
     */
    public YtFieldMapper(JooqReaderWithSupplier<M> jooqReader, Table<R> table) {
        this.jooqReader = jooqReader;
        this.table = table;
    }

    /**
     * Получить список полей, для которого работает маппер. В возвращаемые поля уже добавлен алиас, для их определения
     * при конвертации
     */
    @SuppressWarnings("squid:S1452") // Мы действительно не можем знать класс-параметр Field
    public Collection<Field<?>> getFieldsToRead() {
        return jooqReader.getFieldsToRead().stream()
                .map(this::fieldAlias)
                .collect(Collectors.toList());
    }

    /**
     * Получить список полей, для которого работает маппер, полученных через агрегирующую функцию first.
     * В возвращаемые поля уже добавлен алиас, для их определения при конвертации
     */
    public Collection<Field<?>> getFieldsToReadForFirst() {
        return jooqReader.getFieldsForFirstToRead().stream()
                .map(this::fieldAliasForFirst)
                .collect(Collectors.toList());
    }

    public <T> Field<T> fieldAlias(Field<T> field) {
        return field.as(field.getName());
    }

    private <T> Field<T> fieldAliasForFirst(Field<T> field) {
        return YtDSL.first(field).as(field.getName());
    }

    /**
     * Получить модель, со значениями, смапленными из {@code node}
     *
     * @param node нода, полученная из YT
     */
    public M fromNode(YTreeMapNode node) {
        YtRecord record = new YtRecord<>(table);

        for (Field<?> field : record.fields()) {
            YTreeNode nodeValue = node.get(field.getName()).orElse(YTree.entityNode());

            Object value = extractValue(field, nodeValue);
            //noinspection unchecked
            record.set((Field<Object>) field, value);
        }

        return jooqReader.fromDb(record);
    }

    /**
     * Получить список моделей, со значениями, смапленными из набора строк {@code rowset}
     *
     * @param rowset {@link UnversionedRowset}, полученный из YT
     */
    public List<M> fromRowset(UnversionedRowset rowset) {
        TableSchema schema = rowset.getSchema();

        Map<Field<?>, Integer> fieldToIdx = new HashMap<>();

        for (Field<?> field : table.fields()) {
            String fieldName = field.getName();
            int idx = schema.findColumn(fieldName);
            fieldToIdx.put(field, idx);
        }

        List<M> result = new ArrayList<>(rowset.getRows().size());
        for (UnversionedRow row : rowset.getRows()) {
            YtRecord record = new YtRecord<>(table);
            for (Field<?> field : record.fields()) {
                Integer fieldIdx = fieldToIdx.get(field);
                if (fieldIdx == -1) {
                    record.set(field, null);
                } else {
                    UnversionedValue value = row.getValues().get(fieldIdx);

                    //noinspection unchecked
                    record.set((Field<Object>) field, extractValue(field, value));
                }
            }
            M model = jooqReader.fromDb(record);
            result.add(model);
        }
        return result;
    }

    private static Object extractValue(Field<?> field, UnversionedValue value) {
        if (value.getType() == ColumnValueType.NULL) {
            // если value типа NULL, то мы не можем из него извлечь что-то другое
            return null;
        }

        Class<?> type = field.getType();
        if (Integer.class.isAssignableFrom(type)) {
            throw new IllegalArgumentException(String.format("Field type %s is not supported", type));
        } else if (ULong.class.isAssignableFrom(type)) {
            return ULong.valueOf(value.longValue());
        } else if (Long.class.isAssignableFrom(type)) {
            // Некоторые long'и на самом деле boolean'ы. Это решается в "DIRECT-75542: Научиться генерировать Boolean
            // поля по YT-таблицам"
            if (value.getType() == ColumnValueType.BOOLEAN) {
                return value.booleanValue() ? 1L : 0L;
            }
            return value.longValue();
        } else if (String.class.isAssignableFrom(type)) {
            return value.stringValue();
        } else if (Double.class.isAssignableFrom(type)) {
            return value.doubleValue();
        } else if (Float.class.isAssignableFrom(type)) {
            throw new IllegalArgumentException(String.format("Field type %s is not supported", type));
        } else if (Boolean.class.isAssignableFrom(type)) {
            logger.warn("Unusual parameter Field of type {} in extractValue(..)", type);
            return value.booleanValue();
        } else if (type.isEnum()) {
            //noinspection unchecked
            return Enum.valueOf((Class) type, value.stringValue());
        } else {
            return null;
        }
    }

    private <V> Object extractValue(Field<V> field, @Nullable YTreeNode node) {
        if (node == null || node.isEntityNode()) {
            return null;
        }

        Class<V> type = field.getType();
        if (Integer.class.isAssignableFrom(type)) {
            return node.intValue();
        } else if (ULong.class.isAssignableFrom(type)) {
            return ULong.valueOf(node.longValue());
        } else if (Long.class.isAssignableFrom(type)) {
            // Некоторые long'и на самом деле boolean'ы. Это решается в "DIRECT-75542: Научиться генерировать Boolean
            // поля по YT-таблицам"
            if (node.isBooleanNode()) {
                return node.boolValue() ? 1L : 0L;
            }
            return node.longValue();
        } else if (String.class.isAssignableFrom(type)) {
            return node.stringValue();
        } else if (Double.class.isAssignableFrom(type)) {
            return node.doubleValue();
        } else if (BigDecimal.class.isAssignableFrom(type)) {
            return BigDecimal.valueOf(node.doubleValue());
        } else if (Float.class.isAssignableFrom(type)) {
            logger.warn("Unusual parameter Field of type {} in extractValue(..)", type);
            return node.doubleValue();
        } else if (Boolean.class.isAssignableFrom(type)) {
            logger.warn("Unusual parameter Field of type {} in extractValue(..)", type);
            return node.boolValue() ? 1L : 0L;
        } else if (type.isEnum()) {
            //noinspection unchecked
            return Enum.valueOf((Class) type, node.stringValue());
        }
        return null;
    }

    private static class YtRecord<R extends TableRecord<R>> extends CustomRecord<R> {
        YtRecord(Table<R> table) {
            super(table);
        }
    }
}
