package ru.yandex.direct.grid.core.util.yt.mapping;

import java.math.BigDecimal;
import java.util.Collection;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.base.Functions;
import one.util.streamex.StreamEx;
import org.jooq.Field;
import org.jooq.types.ULong;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.jooqmapper.read.JooqReaderWithSupplier;
import ru.yandex.direct.model.Model;
import ru.yandex.direct.ytwrapper.dynamic.dsl.YtDSL;
import ru.yandex.inside.yt.kosher.ytree.YTreeMapNode;
import ru.yandex.inside.yt.kosher.ytree.YTreeNode;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

/**
 * Надстройка над {@link JooqReaderWithSupplier}, которая позволяет конвертировать
 * ноды, полученные из YT после выполнения запросов к динамическим таблицам, во внутренние модели приложения.
 *
 * @param <M> тип внутренней модели
 */
@ParametersAreNonnullByDefault
public class YtReader<M extends Model> {
    private static final Logger logger = LoggerFactory.getLogger(YtReader.class);

    private final JooqReaderWithSupplier<M> jooqReader;

    /**
     * @param jooqReader настроенный JooqReader
     */
    public YtReader(JooqReaderWithSupplier<M> jooqReader) {
        this.jooqReader = jooqReader;
    }

    /**
     * @return список полей БД, которые необходимо прочитать для заполнения
     * всех свойств модели, которые умеет читать данный маппер.
     * К возвращаемым полям уже добавлен alias для их определения при конвертации.
     */
    public Collection<Field<?>> getFieldsToRead() {
        return mapList(jooqReader.getFieldsToRead(), this::aliasedField);
    }

    /**
     * @return список полей БД, которые необходимо прочитать через агрегирующую функцию first для заполнения
     * всех свойств модели, которые умеет читать данный маппер.
     * К возвращаемым полям уже добавлен alias для их определения при конвертации.
     */
    public Collection<Field<?>> getFieldsToReadForFirst() {
        return mapList(jooqReader.getFieldsForFirstToRead(), this::aliasedFieldForFirst);
    }

    public <T> Field<T> aliasedField(Field<T> field) {
        return field.as(fieldAlias(field));
    }

    private <T> Field<T> aliasedFieldForFirst(Field<T> field) {
        return YtDSL.first(field).as(fieldAlias(field));
    }

    private <T> String fieldAlias(Field<T> field) {
        return String.join("_", field.getQualifiedName().getName());
    }

    /**
     * Заполнение экземпляра модели прочитанными из БД данными в виде {@link YTreeMapNode}.
     * За получение экземпляра модели отвечает {@code JooqReaderWithSupplier<M> jooqReader},
     * переданный при инициализации.
     * <p>
     * Заполняет только те поля модели, для которых достаточно данных
     * в переданном экземпляре {@link YTreeMapNode}.
     * <p>
     * Если ни одно поле модели не было прочитано, генерируется исключение {@link IllegalStateException}.
     *
     * @param row результат чтения из YT
     * @return экземпляр модели, заполненный данными из базы
     */
    public M fromYTreeRow(YTreeMapNode row) {
        var nodeByField = StreamEx.of(jooqReader.getFieldsToRead())
                .append(jooqReader.getFieldsForFirstToRead())
                .mapToEntry(Functions.compose(row::get, this::fieldAlias))
                .filterValues(Optional::isPresent)
                .mapValues(Optional::get)
                .toMap();

        var record = YtDSL.ytContext().newRecord(nodeByField.keySet());
        for (var entry : nodeByField.entrySet()) {
            @SuppressWarnings("unchecked")
            var field = (Field<Object>) entry.getKey();
            record.set(field, extractValue(field, entry.getValue()));
        }

        return jooqReader.fromDb(record);
    }

    private <T> Object extractValue(Field<T> field, @Nullable YTreeNode node) {
        if (node == null || node.isEntityNode()) {
            return null;
        }

        Class<T> type = field.getType();
        if (type.isAssignableFrom(Long.class)) {
            // Boolean колонки пока что представляются как Long: DIRECT-75542
            if (node.isBooleanNode()) {
                return node.boolValue() ? 1L : 0L;
            }
            return node.longValue();
        } else if (type.isAssignableFrom(Integer.class)) {
            return node.intValue();
        } else if (type.isAssignableFrom(ULong.class)) {
            return ULong.valueOf(node.longValue());
        } else if (type.isAssignableFrom(BigDecimal.class)) {
            if (node.isIntegerNode()) {
                return BigDecimal.valueOf(node.longValue());
            }
            return BigDecimal.valueOf(node.doubleValue());
        } else if (type.isAssignableFrom(Double.class)) {
            return node.doubleValue();
        } else if (type.isAssignableFrom(Float.class)) {
            logger.warn("unusual field type '{}' in extractValue(...)", type);
            return node.floatValue();
        } else if (type.isAssignableFrom(Boolean.class)) {
            logger.warn("unusual field type '{}' in extractValue(...)", type);
            return node.boolValue();
        } else if (type.isAssignableFrom(String.class)) {
            return node.stringValue();
        } else if (type.isEnum()) {
            @SuppressWarnings({"rawtypes", "unchecked"})
            Enum value = Enum.valueOf((Class<Enum>) type, node.stringValue());
            return value;
        }
        logger.warn("unusual field type '{}' in extractValue(...)", type);
        return null;
    }
}
