package ru.yandex.direct.grid.model.entity.adgroup;

import java.util.List;
import java.util.Optional;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.core.entity.adgroup.model.AdGroupType;
import ru.yandex.direct.core.entity.pricepackage.model.PricePackageBase;
import ru.yandex.direct.core.entity.product.model.ProductType;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;

import static java.util.Arrays.asList;
import static java.util.Collections.singletonList;

@ParametersAreNonnullByDefault
public class AdGroupTypeConverter {

    // в мапу складываем типы которые соотносятся 1 к 1 AdGroupType <-> GdAdGroupType
    // некоторые AdGroupType конвертируются в GdAdGroupType в зависимости от типа кампании
    // такие вычисления делаем непосредственно в методах конвертации
    private static final BiMap<AdGroupType, GdAdGroupType> TO_GD_AD_GROUP_TYPE_MAP =
            ImmutableBiMap.<AdGroupType, GdAdGroupType>builder()
                    .put(AdGroupType.BASE, GdAdGroupType.TEXT)
                    .put(AdGroupType.DYNAMIC, GdAdGroupType.DYNAMIC)
                    .put(AdGroupType.MOBILE_CONTENT, GdAdGroupType.MOBILE_CONTENT)
                    .put(AdGroupType.PERFORMANCE, GdAdGroupType.PERFORMANCE)
                    .put(AdGroupType.MCBANNER, GdAdGroupType.MCBANNER)
                    .put(AdGroupType.CPM_BANNER, GdAdGroupType.CPM_BANNER)
                    .put(AdGroupType.CPM_VIDEO, GdAdGroupType.CPM_VIDEO)
                    .put(AdGroupType.CPM_AUDIO, GdAdGroupType.CPM_AUDIO)
                    .put(AdGroupType.CPM_INDOOR, GdAdGroupType.CPM_INDOOR)
                    .put(AdGroupType.CPM_OUTDOOR, GdAdGroupType.CPM_OUTDOOR)
                    .put(AdGroupType.INTERNAL, GdAdGroupType.INTERNAL)
                    .put(AdGroupType.CPM_GEOPRODUCT, GdAdGroupType.CPM_GEOPRODUCT)
                    .put(AdGroupType.CPM_GEO_PIN, GdAdGroupType.CPM_GEO_PIN)
                    .put(AdGroupType.CPM_YNDX_FRONTPAGE, GdAdGroupType.CPM_YNDX_FRONTPAGE)
                    .put(AdGroupType.CONTENT_PROMOTION, GdAdGroupType.CONTENT_PROMOTION)
                    .build();
    private static final BiMap<GdAdGroupType, AdGroupType> GD_AD_GROUP_TYPE_TO_INTERNAL_TYPE_MAP =
            TO_GD_AD_GROUP_TYPE_MAP.inverse();

    private static final Set<GdAdGroupType> CONTENT_PROMOTION_ADGROUP_TYPES =
            ImmutableSet.of(GdAdGroupType.CONTENT_PROMOTION_VIDEO, GdAdGroupType.CONTENT_PROMOTION_COLLECTION,
                    GdAdGroupType.CONTENT_PROMOTION_SERVICE, GdAdGroupType.CONTENT_PROMOTION_EDA);

    public static GdAdGroupType toGdAdGroupType(AdGroupType adGroupType, GdCampaignType campaignType) {
        if (adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE && campaignType == GdCampaignType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE;
        }
        if (adGroupType == AdGroupType.CPM_VIDEO && campaignType == GdCampaignType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE_VIDEO;
        }
        if (adGroupType == AdGroupType.CPM_AUDIO && campaignType == GdCampaignType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE_AUDIO;
        }
        if (adGroupType == AdGroupType.CPM_BANNER && campaignType == GdCampaignType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE_BANNER;
        }
        return TO_GD_AD_GROUP_TYPE_MAP.get(adGroupType);
    }

    /**
     * Версия метода, которая поддерживает синтетические типы групп с учётом пакета
     */
    public static GdAdGroupType toGdAdGroupType(AdGroupType adGroupType, GdCampaignType campaignType,
                                                @Nullable PricePackageBase pricePackage) {
        //если это прайсовое видео для главной, то отдельный тип группы
        if (adGroupType == AdGroupType.CPM_VIDEO && campaignType == GdCampaignType.CPM_PRICE
                && Optional.of(pricePackage).map(PricePackageBase::getIsFrontpage).orElse(false)) {
            return GdAdGroupType.CPM_PRICE_FRONTPAGE_VIDEO;
        }
        return toGdAdGroupType(adGroupType, campaignType);
    }

    /**
     * Версия метода, которая поддерживает синтетические типы групп
     */
    public static List<GdAdGroupType> toGdAdGroupTypes(AdGroupType adGroupType, GdCampaignType campaignType) {
        if (adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE && campaignType == GdCampaignType.CPM_PRICE) {
            return singletonList(GdAdGroupType.CPM_PRICE);
        }
        if (adGroupType == AdGroupType.CPM_VIDEO && campaignType == GdCampaignType.CPM_PRICE) {
            return singletonList(GdAdGroupType.CPM_PRICE_VIDEO);
        }
        if (adGroupType == AdGroupType.CPM_AUDIO && campaignType == GdCampaignType.CPM_PRICE) {
            return singletonList(GdAdGroupType.CPM_PRICE_AUDIO);
        }
        if (adGroupType == AdGroupType.CPM_BANNER && campaignType == GdCampaignType.CPM_PRICE) {
            return singletonList(GdAdGroupType.CPM_PRICE_BANNER);
        }

        if (adGroupType == AdGroupType.CONTENT_PROMOTION) {
            return asList(GdAdGroupType.CONTENT_PROMOTION_VIDEO, GdAdGroupType.CONTENT_PROMOTION_COLLECTION);
        }

        return singletonList(TO_GD_AD_GROUP_TYPE_MAP.get(adGroupType));
    }

    /**
     * Версия метода, которая поддерживает синтетические типы групп с учётом типа продукта
     */
    public static GdAdGroupType toGdAdGroupType(AdGroupType adGroupType, ProductType productType) {
        if (adGroupType == AdGroupType.CPM_YNDX_FRONTPAGE && productType == ProductType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE;
        }
        if (adGroupType == AdGroupType.CPM_VIDEO && productType == ProductType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE_VIDEO;
        }
        if (adGroupType == AdGroupType.CPM_AUDIO && productType == ProductType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE_AUDIO;
        }
        if (adGroupType == AdGroupType.CPM_BANNER && productType == ProductType.CPM_PRICE) {
            return GdAdGroupType.CPM_PRICE_BANNER;
        }

        return TO_GD_AD_GROUP_TYPE_MAP.get(adGroupType);
    }

    public static AdGroupType toInternalAdGroupType(GdAdGroupType gdAdGroupType) {
        if (gdAdGroupType == GdAdGroupType.CPM_PRICE) {
            return AdGroupType.CPM_YNDX_FRONTPAGE;
        }
        if (gdAdGroupType == GdAdGroupType.CPM_PRICE_VIDEO) {
            return AdGroupType.CPM_VIDEO;
        }
        if (gdAdGroupType == GdAdGroupType.CPM_PRICE_AUDIO) {
            return AdGroupType.CPM_AUDIO;
        }
        if (gdAdGroupType == GdAdGroupType.CPM_PRICE_BANNER) {
            return AdGroupType.CPM_BANNER;
        }
        if (CONTENT_PROMOTION_ADGROUP_TYPES.contains(gdAdGroupType)) {
            return AdGroupType.CONTENT_PROMOTION;
        }
        return GD_AD_GROUP_TYPE_TO_INTERNAL_TYPE_MAP.get(gdAdGroupType);
    }

}
