package ru.yandex.direct.grid.model.entity.campaign.converter;

import java.math.BigDecimal;
import java.net.IDN;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Supplier;
import java.util.function.UnaryOperator;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.Sets;
import one.util.streamex.IntStreamEx;
import one.util.streamex.StreamEx;
import org.apache.commons.lang3.StringUtils;
import org.jetbrains.annotations.NotNull;

import ru.yandex.direct.core.entity.campaign.model.CampAimType;
import ru.yandex.direct.core.entity.campaign.model.CampaignAttributionModel;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurer;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurerParamsIas;
import ru.yandex.direct.core.entity.campaign.model.CampaignMeasurerSystem;
import ru.yandex.direct.core.entity.campaign.model.CampaignSource;
import ru.yandex.direct.core.entity.campaign.model.CampaignStatusModerate;
import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.core.entity.campaign.model.ContentLanguage;
import ru.yandex.direct.core.entity.campaign.model.DayBudgetShowMode;
import ru.yandex.direct.core.entity.campaign.model.EshowsRate;
import ru.yandex.direct.core.entity.campaign.model.EshowsSettings;
import ru.yandex.direct.core.entity.campaign.model.EshowsVideoType;
import ru.yandex.direct.core.entity.campaign.model.ImpressionStandardTime;
import ru.yandex.direct.core.entity.campaign.model.InternalCampaignRestrictionType;
import ru.yandex.direct.core.entity.campaign.model.MeaningfulGoal;
import ru.yandex.direct.core.entity.campaign.model.MobileContentInfo;
import ru.yandex.direct.core.entity.campaign.model.PlacementType;
import ru.yandex.direct.core.entity.campaign.model.RfCloseByClickType;
import ru.yandex.direct.core.entity.currency.model.cpmyndxfrontpage.FrontpageCampaignShowType;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppDeviceTypeTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppNetworkTargeting;
import ru.yandex.direct.core.entity.mobileapp.model.MobileAppTracker;
import ru.yandex.direct.currency.Money;
import ru.yandex.direct.grid.model.campaign.GdBroadMatch;
import ru.yandex.direct.grid.model.campaign.GdCampAimType;
import ru.yandex.direct.grid.model.campaign.GdCampaign;
import ru.yandex.direct.grid.model.campaign.GdCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdCampaignAttributionModel;
import ru.yandex.direct.grid.model.campaign.GdCampaignMeasurer;
import ru.yandex.direct.grid.model.campaign.GdCampaignMeasurerSystem;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlacementType;
import ru.yandex.direct.grid.model.campaign.GdCampaignPlatform;
import ru.yandex.direct.grid.model.campaign.GdCampaignServicedState;
import ru.yandex.direct.grid.model.campaign.GdCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdCampaignStatus;
import ru.yandex.direct.grid.model.campaign.GdCampaignStatusModerate;
import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdContentLanguage;
import ru.yandex.direct.grid.model.campaign.GdContentPromotionCampaign;
import ru.yandex.direct.grid.model.campaign.GdCpmBannerCampaign;
import ru.yandex.direct.grid.model.campaign.GdCpmDealsCampaign;
import ru.yandex.direct.grid.model.campaign.GdCpmYndxFrontpageCampaign;
import ru.yandex.direct.grid.model.campaign.GdCpmYndxFrontpageCampaignShowType;
import ru.yandex.direct.grid.model.campaign.GdDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.GdDynamicCampaign;
import ru.yandex.direct.grid.model.campaign.GdEshowsRate;
import ru.yandex.direct.grid.model.campaign.GdEshowsSettings;
import ru.yandex.direct.grid.model.campaign.GdEshowsVideoType;
import ru.yandex.direct.grid.model.campaign.GdImpressionStandardTime;
import ru.yandex.direct.grid.model.campaign.GdInternalAutobudgetCampaign;
import ru.yandex.direct.grid.model.campaign.GdInternalCampaign;
import ru.yandex.direct.grid.model.campaign.GdInternalCampaignImpressionFrequencyControl;
import ru.yandex.direct.grid.model.campaign.GdInternalDistribCampaign;
import ru.yandex.direct.grid.model.campaign.GdInternalFreeCampaign;
import ru.yandex.direct.grid.model.campaign.GdInternalFreeCampaignRestrictionType;
import ru.yandex.direct.grid.model.campaign.GdMcBannerCampaign;
import ru.yandex.direct.grid.model.campaign.GdMeaningfulGoal;
import ru.yandex.direct.grid.model.campaign.GdMetaCampaignFilterType;
import ru.yandex.direct.grid.model.campaign.GdMobileAppTracker;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaign;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaignDeviceTypeTargeting;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaignNetworkTargeting;
import ru.yandex.direct.grid.model.campaign.GdMobileContentCampaignOsType;
import ru.yandex.direct.grid.model.campaign.GdMobileContentInfo;
import ru.yandex.direct.grid.model.campaign.GdPriceCampaign;
import ru.yandex.direct.grid.model.campaign.GdRfCloseByClick;
import ru.yandex.direct.grid.model.campaign.GdSmartCampaign;
import ru.yandex.direct.grid.model.campaign.GdTextCampaign;
import ru.yandex.direct.grid.model.campaign.GdTrackingSystem;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignAction;
import ru.yandex.direct.grid.model.campaign.GdiCampaignMetatype;
import ru.yandex.direct.grid.model.campaign.GdiCampaignSource;
import ru.yandex.direct.grid.model.campaign.GdiContentLanguage;
import ru.yandex.direct.grid.model.campaign.GdiDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.GdiInternalCampaignRestrictionType;
import ru.yandex.direct.grid.model.campaign.facelift.GdBudgetDisplayFormat;
import ru.yandex.direct.grid.model.campaign.facelift.GdCampaignAdditionalData;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.campaign.timetarget.GdHolidaysSettings;
import ru.yandex.direct.grid.model.campaign.timetarget.GdTimeTarget;
import ru.yandex.direct.grid.processing.model.recommendation.GdRecommendation;
import ru.yandex.direct.libs.timetarget.HoursCoef;
import ru.yandex.direct.libs.timetarget.TimeTarget;
import ru.yandex.direct.libs.timetarget.WeekdayType;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Collections.emptySet;
import static java.util.Map.entry;
import static org.joda.time.DateTimeConstants.DAYS_PER_WEEK;
import static org.joda.time.DateTimeConstants.HOURS_PER_DAY;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.DISABLED_DOMAINS_DELIMITER;
import static ru.yandex.direct.core.entity.campaign.repository.CampaignMappings.DISABLED_IPS_DELIMITER;
import static ru.yandex.direct.grid.model.entity.campaign.converter.GdCampaignFeatureHelper.isCampaignAimingAllowed;
import static ru.yandex.direct.grid.model.entity.campaign.converter.PriceFlightDataConverter.toGdPriceFlightReasonIncorrect;
import static ru.yandex.direct.grid.model.entity.campaign.converter.PriceFlightDataConverter.toGdPriceFlightStatusApprove;
import static ru.yandex.direct.grid.model.entity.campaign.converter.PriceFlightDataConverter.toGdPriceFlightStatusCorrect;
import static ru.yandex.direct.grid.model.entity.campaign.converter.PriceFlightDataConverter.toGdPriceFlightTargetingsSnapshot;
import static ru.yandex.direct.grid.model.utils.RfConverter.toGdRfPeriod;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;
import static ru.yandex.direct.utils.FunctionalUtils.mapSet;
import static ru.yandex.direct.utils.JsonUtils.fromJson;

@ParametersAreNonnullByDefault
public class CampaignDataConverter {
    public static final String WWW = "www";
    private static final Map<CampAimType, GdCampAimType> CAMP_AIM_TYPE_CORE_TO_GD =
            Arrays.stream(GdCampAimType.values()).collect(Collectors.toMap(GdCampAimType::getTypedValue, e -> e));

    private static final BiMap<CampaignType, GdCampaignType> TO_GD_CAMPAIGN_TYPE_MAP =
            ImmutableBiMap.<CampaignType, GdCampaignType>builder()
                    .put(CampaignType.TEXT, GdCampaignType.TEXT)
                    .put(CampaignType.DYNAMIC, GdCampaignType.DYNAMIC)
                    .put(CampaignType.MOBILE_CONTENT, GdCampaignType.MOBILE_CONTENT)
                    .put(CampaignType.PERFORMANCE, GdCampaignType.PERFORMANCE)
                    .put(CampaignType.MCBANNER, GdCampaignType.MCBANNER)
                    .put(CampaignType.CPM_DEALS, GdCampaignType.CPM_DEALS)
                    .put(CampaignType.CPM_BANNER, GdCampaignType.CPM_BANNER)
                    .put(CampaignType.CPM_YNDX_FRONTPAGE, GdCampaignType.CPM_YNDX_FRONTPAGE)
                    .put(CampaignType.INTERNAL_AUTOBUDGET, GdCampaignType.INTERNAL_AUTOBUDGET)
                    .put(CampaignType.INTERNAL_DISTRIB, GdCampaignType.INTERNAL_DISTRIB)
                    .put(CampaignType.INTERNAL_FREE, GdCampaignType.INTERNAL_FREE)
                    .put(CampaignType.CONTENT_PROMOTION, GdCampaignType.CONTENT_PROMOTION)
                    .put(CampaignType.CPM_PRICE, GdCampaignType.CPM_PRICE)
                    .build();
    private static final Map<GdCampaignType, CampaignType> GD_CAMPAIGN_TYPE_TO_INTERNAL_TYPE_MAP =
            TO_GD_CAMPAIGN_TYPE_MAP.inverse();

    private static final Map<GdiInternalCampaignRestrictionType, GdInternalFreeCampaignRestrictionType>
            TO_GD_INTERNAL_FREE_CAMPAIGN_RESTRICTION_TYPE_MAP = Map.ofEntries(
            entry(GdiInternalCampaignRestrictionType.CLICKS, GdInternalFreeCampaignRestrictionType.CLICKS),
            entry(GdiInternalCampaignRestrictionType.SHOWS, GdInternalFreeCampaignRestrictionType.SHOWS),
            entry(GdiInternalCampaignRestrictionType.DAYS, GdInternalFreeCampaignRestrictionType.DAYS)
    );

    private static final BiMap<RfCloseByClickType, GdRfCloseByClick> TO_GD_RF_CLOSE_BY_CLICK_TYPE_MAP =
            ImmutableBiMap.<RfCloseByClickType, GdRfCloseByClick>builder()
                    .put(RfCloseByClickType.CAMPAIGN, GdRfCloseByClick.ON_CAMPAIGN)
                    .put(RfCloseByClickType.ADGROUP, GdRfCloseByClick.ON_AD_GROUP)
                    .build();

    private static final Map<GdRfCloseByClick, RfCloseByClickType> TO_RF_CLOSE_BY_CLICK_TYPE_MAP =
            TO_GD_RF_CLOSE_BY_CLICK_TYPE_MAP.inverse();

    private CampaignDataConverter() {
    }

    public static GdCampaignType toGdCampaignType(CampaignType campaignType) {
        return TO_GD_CAMPAIGN_TYPE_MAP.get(campaignType);
    }

    public static CampaignType toCampaignType(GdCampaignType gdCampaignType) {
        return GD_CAMPAIGN_TYPE_TO_INTERNAL_TYPE_MAP.get(gdCampaignType);
    }

    public static GdCampaignPlatform toGdCampaignPlatform(CampaignsPlatform campaignPlatform) {
        return GdCampaignPlatform.valueOf(campaignPlatform.name());
    }

    public static CampaignsPlatform toCampaignsPlatform(GdCampaignPlatform gdCampaignPlatform) {
        return CampaignsPlatform.valueOf(gdCampaignPlatform.name());
    }

    public static GdCampaignStatusModerate toGdCampaignStatusModerate(CampaignStatusModerate campaignStatusModerate) {
        return GdCampaignStatusModerate.valueOf(campaignStatusModerate.name());
    }

    public static GdCampaignAction toGdCampaignAction(GdiCampaignAction campaignAccess) {
        return GdCampaignAction.valueOf(campaignAccess.name());
    }

    public static GdCampaignSource toGdCampaignSource(GdiCampaignSource source) {
        return GdCampaignSource.valueOf(source.name());
    }

    public static GdCampaignSource toGdCampaignSource(CampaignSource source) {
        return GdCampaignSource.valueOf(source.name());
    }

    public static GdCampaign toGdCampaignImplementation(GdiCampaign internal, GdCampaignStatus campaignStatus,
                                                        GdCampaignFlatStrategy campaignFlatStrategy,
                                                        GdCampaignFlatStrategy campaignStrategy,
                                                        List<GdRecommendation> recommendations,
                                                        UnaryOperator<List<Long>> geoToFrontendGeo,
                                                        boolean advancedGeoTargeting) {
        GdCampaign typedCampaign;
        switch (internal.getType()) {
            case TEXT:
                typedCampaign = new GdTextCampaign()
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withContactInfo(internal.getContactInfo())
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withHasTitleSubstitution(internal.getHasTitleSubstitution())
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withIsAloneTrafaretAllowed(internal.getIsAloneTrafaretAllowed())
                        .withRequireFiltrationByDontShowDomains(internal.getRequireFiltrationByDontShowDomains())
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withHasAddOpenstatTagToUrl(internal.getHasAddOpenstatTagToUrl())
                        .withClientDialogId(internal.getClientDialogId())
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withEnableCompanyInfo(internal.getEnableCompanyInfo())
                        .withExcludePausedCompetingAds(internal.getExcludePausedCompetingAds())
                        .withBroadMatch(toGdBroadMatch(internal))
                        .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                        .withDayBudget(internal.getDayBudget())
                        .withDayBudgetShowMode(toGdDayBudgetShowMode(internal.getDayBudgetShowMode()))
                        .withDayBudgetDailyChangeCount(internal.getDayBudgetDailyChangeCount())
                        .withAllowedPageIds(internal.getAllowedPageIds())
                        .withAllowedDomains(toAllowedDomains(internal.getAllowedDomains(), internal.getAllowedSsp()))
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withPlacementTypes(toGdPlacementTypes(internal.getPlacementTypes()))
                        .withTurboAppsEnabled(internal.getTurboAppsEnabled())
                        .withDefaultPermalinkId(internal.getDefaultPermalinkId())
                        .withDefaultTrackingPhoneId(internal.getDefaultTrackingPhoneId())
                        .withDefaultChainId(internal.getDefaultChainId())
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withIsUniversalCamp(internal.getIsUniversal())
                        .withIsGoodsCamp(internal.getIsGoods())
                        .withIsMeaningfulGoalsValuesFromMetrikaEnabled(internal.getIsMeaningfulGoalsValuesFromMetrikaEnabled())
                        .withCampAimType(CAMP_AIM_TYPE_CORE_TO_GD.get(internal.getCampAimType()))
                        .withCalltrackingSettingsId(internal.getCalltrackingSettingsId())
                        .withPromoExtensionId(internal.getPromoExtensionId())
                        .withIsSimplifiedStrategyViewEnabled(internal.getIsSimplifiedStrategyViewEnabled())
                        .withIsOrderPhraseLengthPrecedenceEnabled(internal.getIsOrderPhraseLengthPrecedenceEnabled())
                        .withIsAllowedOnAdultContent(internal.getIsAllowedOnAdultContent())
                        .withIsS2sTrackingEnabled(internal.getIsS2sTrackingEnabled())
                        .withStrategyId(getEffectiveStrategyId(internal))
                        .withBannerHrefParams(internal.getBannerHrefParams());
                break;
            case MOBILE_CONTENT:
                typedCampaign = new GdMobileContentCampaign()
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withIsAloneTrafaretAllowed(internal.getIsAloneTrafaretAllowed())
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withAllowedPageIds(internal.getAllowedPageIds())
                        .withAllowedDomains(toAllowedDomains(internal.getAllowedDomains(), internal.getAllowedSsp()))
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withDayBudget(internal.getDayBudget())
                        .withDayBudgetShowMode(toGdDayBudgetShowMode(internal.getDayBudgetShowMode()))
                        .withDayBudgetDailyChangeCount(internal.getDayBudgetDailyChangeCount())
                        .withDeviceTypeTargeting(toGdMobileContentCampaignDeviceTypeTargeting(internal.getDeviceTypeTargeting()))
                        .withNetworkTargeting(toGdMobileContentCampaignNetworkTargeting(internal.getNetworkTargeting()))
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withIsInstalledApp(internal.getIsInstalledApp())
                        .withMobileContentInfo(toGdMobileContentInfo(internal.getMobileContentInfo()))
                        .withMobileAppId(internal.getMobileAppId())
                        .withIsOrderPhraseLengthPrecedenceEnabled(internal.getIsOrderPhraseLengthPrecedenceEnabled())
                        .withIsSkadNetworkEnabled(internal.getIsSkadNetworkEnabled())
                        .withIsNewIosVersionEnabled(internal.getIsNewIosVersionEnabled())
                        .withStrategyId(getEffectiveStrategyId(internal));
                break;
            case DYNAMIC:
                typedCampaign = new GdDynamicCampaign()
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withContactInfo(internal.getContactInfo())
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withHasTitleSubstitution(internal.getHasTitleSubstitution())
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withIsAloneTrafaretAllowed(internal.getIsAloneTrafaretAllowed())
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withHasAddOpenstatTagToUrl(internal.getHasAddOpenstatTagToUrl())
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withRequireFiltrationByDontShowDomains(internal.getRequireFiltrationByDontShowDomains())
                        .withEnableCompanyInfo(internal.getEnableCompanyInfo())
                        .withBroadMatch(toGdBroadMatch(internal))
                        .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                        .withDayBudget(internal.getDayBudget())
                        .withDayBudgetShowMode(toGdDayBudgetShowMode(internal.getDayBudgetShowMode()))
                        .withDayBudgetDailyChangeCount(internal.getDayBudgetDailyChangeCount())
                        .withAllowedPageIds(internal.getAllowedPageIds())
                        .withAllowedDomains(toAllowedDomains(internal.getAllowedDomains(), internal.getAllowedSsp()))
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withPlacementTypes(toGdPlacementTypes(internal.getPlacementTypes()))
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withDefaultPermalinkId(internal.getDefaultPermalinkId())
                        .withPromoExtensionId(internal.getPromoExtensionId())
                        .withIsSimplifiedStrategyViewEnabled(internal.getIsSimplifiedStrategyViewEnabled())
                        .withIsOrderPhraseLengthPrecedenceEnabled(internal.getIsOrderPhraseLengthPrecedenceEnabled())
                        .withIsMeaningfulGoalsValuesFromMetrikaEnabled(internal.getIsMeaningfulGoalsValuesFromMetrikaEnabled())
                        .withIsS2sTrackingEnabled(internal.getIsS2sTrackingEnabled())
                        .withStrategyId(getEffectiveStrategyId(internal))
                        .withBannerHrefParams(internal.getBannerHrefParams());
                break;
            case PERFORMANCE:
                typedCampaign = new GdSmartCampaign()
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withIsAloneTrafaretAllowed(internal.getIsAloneTrafaretAllowed())
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withHasTurboSmarts(internal.getHasTurboSmarts())
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withBroadMatch(toGdBroadMatch(internal))
                        .withTurboAppsEnabled(internal.getTurboAppsEnabled())
                        .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                        .withRequireFiltrationByDontShowDomains(internal.getRequireFiltrationByDontShowDomains())
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withIsMeaningfulGoalsValuesFromMetrikaEnabled(internal.getIsMeaningfulGoalsValuesFromMetrikaEnabled())
                        .withIsSimplifiedStrategyViewEnabled(internal.getIsSimplifiedStrategyViewEnabled())
                        .withIsS2sTrackingEnabled(internal.getIsS2sTrackingEnabled())
                        .withStrategyId(getEffectiveStrategyId(internal))
                        .withBannerHrefParams(internal.getBannerHrefParams())
                        .withIsWwManagedOrder(internal.getIsWwManagedOrder());
                break;
            case MCBANNER:
                typedCampaign = new GdMcBannerCampaign()
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withHasAddOpenstatTagToUrl(internal.getHasAddOpenstatTagToUrl())
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withDayBudget(internal.getDayBudget())
                        .withDayBudgetShowMode(toGdDayBudgetShowMode(internal.getDayBudgetShowMode()))
                        .withDayBudgetDailyChangeCount(internal.getDayBudgetDailyChangeCount())
                        .withIsOrderPhraseLengthPrecedenceEnabled(internal.getIsOrderPhraseLengthPrecedenceEnabled())
                        .withStrategyId(getEffectiveStrategyId(internal));
                break;
            case CPM_DEALS:
                typedCampaign = new GdCpmDealsCampaign()
                        .withAllowedPageIds(internal.getAllowedPageIds())
                        .withAllowedDomains(toAllowedDomains(internal.getAllowedDomains(), internal.getAllowedSsp()))
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting());
                break;
            case CPM_BANNER:
                typedCampaign = new GdCpmBannerCampaign()
                        .withDisabledPlaces(toDisabledPlacesSet(internal.getDisabledDomains(),
                                internal.getDisabledSsp()))
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withRequireFiltrationByDontShowDomains(internal.getRequireFiltrationByDontShowDomains())
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withHasAddOpenstatTagToUrl(internal.getHasAddOpenstatTagToUrl())
                        .withAllowedPageIds(internal.getAllowedPageIds())
                        .withDisallowedPageIds(internal.getDisallowedPageIds())
                        .withAllowedDomains(toAllowedDomains(internal.getAllowedDomains(), internal.getAllowedSsp()))
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withBrandSurveyId(internal.getBrandSurveyId())
                        .withBrandSurveyName(internal.getBrandSurveyName())
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withDisabledVideoAdsPlaces(internal.getDisabledVideoPlacements())
                        .withImpressionRateCount(internal.getImpressionRateCount())
                        .withImpressionRateIntervalDays(internal.getImpressionRateIntervalDays())
                        .withEshowsSettings(toGdEshowsSettings(internal.getEshowsSettings()))
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withImpressionStandardTime(toGdImpressionStandardTime(internal.getImpressionStandardTime()))
                        .withIsAllowedOnAdultContent(internal.getIsAllowedOnAdultContent())
                        .withStrategyId(getEffectiveStrategyId(internal));

                break;

            case INTERNAL_AUTOBUDGET:
                typedCampaign = getTypedInternalCampaignWithCommonFields(GdInternalAutobudgetCampaign::new, internal)
                        .withStrategyId(getEffectiveStrategyId(internal));
                break;
            case INTERNAL_DISTRIB:
                typedCampaign = getTypedInternalCampaignWithCommonFields(GdInternalDistribCampaign::new, internal)
                        .withRotationGoalId(internal.getInternalAdRotationGoalId());
                break;
            case INTERNAL_FREE:
                typedCampaign = getTypedInternalCampaignWithCommonFields(GdInternalFreeCampaign::new, internal)
                        .withRestrictionType(
                                toGdInternalFreeCampaignRestrictionType(internal.getInternalAdRestrictionType()))
                        .withRestrictionValue(internal.getInternalAdRestrictionValue());
                break;

            case CONTENT_PROMOTION:
                typedCampaign = new GdContentPromotionCampaign()
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withHasExtendedGeoTargeting(internal.getHasExtendedGeoTargeting())
                        .withUseCurrentRegion(
                                getUseCurrentRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        advancedGeoTargeting)
                        )
                        .withUseRegularRegion(
                                getUseRegularRegion(
                                        internal.getUseCurrentRegion(),
                                        internal.getUseRegularRegion(),
                                        internal.getHasExtendedGeoTargeting(),
                                        advancedGeoTargeting
                                )
                        )
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withDayBudget(internal.getDayBudget())
                        .withDayBudgetShowMode(toGdDayBudgetShowMode(internal.getDayBudgetShowMode()))
                        .withStrategyId(getEffectiveStrategyId(internal));
                break;
            case CPM_YNDX_FRONTPAGE:
                typedCampaign = new GdCpmYndxFrontpageCampaign()
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withHasAddOpenstatTagToUrl(internal.getHasAddOpenstatTagToUrl())
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withAllowedFrontpageType(toGdCpmYndxFrontpageCampaignShowType(internal.getAllowedFrontpageTypes()))
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withBrandSurveyId(internal.getBrandSurveyId())
                        .withBrandSurveyName(internal.getBrandSurveyName())
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withImpressionStandardTime(toGdImpressionStandardTime(internal.getImpressionStandardTime()))
                        .withStrategyId(getEffectiveStrategyId(internal));
                break;
            case CPM_PRICE:
                typedCampaign = new GdPriceCampaign()
                        .withMetrikaCounters(internal.getMetrikaCounters())
                        .withBrandSurveyId(internal.getBrandSurveyId())
                        .withBrandSurveyName(internal.getBrandSurveyName())
                        .withAbSegmentStatisticRetargetingConditionId(
                                internal.getAbSegmentStatisticRetargetingConditionId())
                        .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                        .withHasAddMetrikaTagToUrl(internal.getHasAddMetrikaTagToUrl())
                        .withHasAddOpenstatTagToUrl(internal.getHasAddOpenstatTagToUrl())
                        .withContentLanguage(toGdContentLanguage(internal.getContentLanguage()))
                        .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                        .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                        .withAutoProlongation(campaignStrategy.getBudget().getAutoProlongation())
                        .withPricePackageId(internal.getPricePackageId())
                        .withDisabledVideoAdsPlaces(internal.getDisabledVideoPlacements())
                        .withDisabledPlaces(
                                toDisabledPlacesSet(internal.getDisabledDomains(), internal.getDisabledSsp())
                        )
                        .withRequireFiltrationByDontShowDomains(internal.getRequireFiltrationByDontShowDomains())
                        .withImpressionRateCount(internal.getImpressionRateCount())
                        .withImpressionRateIntervalDays(internal.getImpressionRateIntervalDays())
                        .withAdditionalData(toGdCampaignAdditionalData(internal))
                        .withFlightTargetingsSnapshot(toGdPriceFlightTargetingsSnapshot(
                                internal.getFlightTargetingsSnapshot(), geoToFrontendGeo))
                        .withFlightOrderVolume(internal.getFlightOrderVolume())
                        .withFlightStatusApprove(toGdPriceFlightStatusApprove(internal.getFlightStatusApprove()))
                        .withFlightStatusCorrect(toGdPriceFlightStatusCorrect(internal.getFlightStatusCorrect()))
                        .withFlightReasonIncorrect(toGdPriceFlightReasonIncorrect(internal.getFlightReasonIncorrect()))
                        .withDisallowedPageIds(internal.getDisallowedPageIds())
                        .withIsAllowedOnAdultContent(internal.getIsAllowedOnAdultContent())
                        .withStrategyId(getEffectiveStrategyId(internal));
                break;
            default:
                throw new IllegalArgumentException("Unsupported campaign type: " + internal.getType());
        }

        return typedCampaign.withId(internal.getId())
                .withWalletId(internal.getWalletId() != 0 ? internal.getWalletId() : null)
                .withName(internal.getName())
                .withAgencyUserId(internal.getAgencyUserId())
                .withAgencyId(internal.getAgencyId() != 0 ? internal.getAgencyId() : null)
                .withManagerUserId(internal.getManagerUserId())
                .withType(toGdCampaignType(internal.getType()))
                .withSum(extractCampaignSum(internal))
                .withSumRest(internal.getSumRest())
                .withNoPay(internal.getNoPay())
                .withCurrency(internal.getCurrencyCode())
                .withStartDate(internal.getStartDate())
                .withShows(internal.getShows())
                .withClicks(internal.getClicks())
                .withEndDate(internal.getFinishDate())
                .withFlatStrategy(campaignFlatStrategy)
                .withStrategy(campaignStrategy)
                .withContextLimit(internal.getContextLimit())
                .withHasEnableCpcHold(internal.getHasEnableCpcHold())
                .withTimeTarget(toGdTimeTarget(internal.getTimeTarget()).withIdTimeZone(internal.getTimezoneId()))
                .withStatus(campaignStatus)
                .withHasAds(internal.getHasBanners())
                .withHasNotArchivedAds(internal.getHasNotArchiveBanners())
                .withRecommendations(recommendations)
                .withIsAimingAllowed(isCampaignAimingAllowed(internal))
                .withPromocodeRestrictedDomain(getPromocodeRestrictedDomainViews(
                        internal.getPromocodeRestrictedDomain()))
                .withDisabledIps(toDisabledIps(internal.getDisabledIps()))
                .withMinusKeywords(internal.getMinusKeywords())
                .withExportId(internal.getOrderId())
                .withDescription(internal.getDescription())
                .withBrandSafetyCategories(internal.getBrandSafetyCategories())
                .withFeatures(GdCampaignFeatureHelper.getCampaignFeatures(internal, campaignStrategy))
                .withSource(toGdCampaignSource(internal.getSource()))
                .withIsRecommendationsManagementEnabled(internal.getIsRecommendationsManagementEnabled())
                .withIsPriceRecommendationsManagementEnabled(internal.getIsPriceRecommendationsManagementEnabled())
                .withCampaignsPromotions(internal.getCampaignsPromotions());
    }

    @NotNull
    public static BigDecimal extractCampaignSum(GdiCampaign internal) {
        BigDecimal campaignSum = Money.valueOf(internal.getSum().subtract(internal.getSumSpent()),
                internal.getCurrencyCode())
                .roundToCentDown()
                .bigDecimalValue();

        return internal.getWalletId() != 0 ? BigDecimal.ZERO : campaignSum;
    }

    private static GdCampaignAdditionalData toGdCampaignAdditionalData(GdiCampaign internal) {
        return new GdCampaignAdditionalData()
                .withHref(internal.getHref())
                .withBudgetDisplayFormat(GdBudgetDisplayFormat.fromSource(internal.getBudgetDisplayFormat()))
                .withCompanyName(internal.getCompanyName())
                .withBusinessCategory(internal.getBusinessCategory());
    }

    private static GdInternalFreeCampaignRestrictionType toGdInternalFreeCampaignRestrictionType(
            GdiInternalCampaignRestrictionType restrictionType) {
        //noinspection ConstantConditions
        checkArgument(restrictionType != null
                        && TO_GD_INTERNAL_FREE_CAMPAIGN_RESTRICTION_TYPE_MAP.containsKey(restrictionType),
                "unexpected restrictionType for InternalFreeCampaign: %s", restrictionType);
        return TO_GD_INTERNAL_FREE_CAMPAIGN_RESTRICTION_TYPE_MAP.get(restrictionType);
    }

    private static <T extends GdInternalCampaign> T getTypedInternalCampaignWithCommonFields(
            Supplier<T> constructor, GdiCampaign internal) {
        GdInternalCampaign gdInternalCampaign = constructor.get()
                .withMetrikaCounters(internal.getMetrikaCounters())
                .withMeaningfulGoals(toGdMeaningFulGoals(internal.getMeaningfulGoals()))
                .withAttributionModel(toGdAttributionModel(internal.getAttributionModel()))
                .withPlaceId(internal.getInternalAdPlaceId())
                .withIsMobile(internal.getInternalAdIsMobile())
                .withPageId(internal.getInternalAdPageId())
                .withAbSegmentStatisticRetargetingConditionId(
                        internal.getAbSegmentStatisticRetargetingConditionId())
                .withAbSegmentRetargetingConditionId(internal.getAbSegmentRetargetingConditionId())
                .withRfCloseByClick(toGdRfCloseByClick(internal.getInternalAdRfCloseByClick()));

        boolean hasShowsRf =
                internal.getImpressionRateCount() != null && internal.getImpressionRateCount() != 0;
        boolean hasClicksRf =
                internal.getInternalAdMaxClicksCount() != null && internal.getInternalAdMaxClicksCount() != 0;
        boolean hasClosesRf =
                internal.getInternalAdMaxStopsCount() != null && internal.getInternalAdMaxStopsCount() != 0;
        if (hasShowsRf || hasClicksRf || hasClosesRf) {
            gdInternalCampaign.setImpressionFrequencyControl(
                    new GdInternalCampaignImpressionFrequencyControl()
                            .withDays(internal.getImpressionRateIntervalDays())
                            .withImpressions(internal.getImpressionRateCount())
                            .withMaxClicksCount(internal.getInternalAdMaxClicksCount())
                            .withMaxClicksPeriod(toGdRfPeriod(internal.getInternalAdMaxClicksPeriod()))
                            .withMaxStopsCount(internal.getInternalAdMaxStopsCount())
                            .withMaxStopsPeriod(toGdRfPeriod(internal.getInternalAdMaxStopsPeriod())));
        }

        @SuppressWarnings("unchecked")
        T typedCampaign = (T) gdInternalCampaign;

        return typedCampaign;
    }

    @Nullable
    private static GdRfCloseByClick toGdRfCloseByClick(@Nullable RfCloseByClickType internalAdRfCloseByClick) {
        if (internalAdRfCloseByClick == null) {
            return null;
        }

        checkArgument(TO_GD_RF_CLOSE_BY_CLICK_TYPE_MAP.containsKey(internalAdRfCloseByClick),
                "unexpected rfCloseByClick for internalCampaign: %s", internalAdRfCloseByClick);
        return TO_GD_RF_CLOSE_BY_CLICK_TYPE_MAP.get(internalAdRfCloseByClick);
    }

    @Nullable
    public static RfCloseByClickType toRfCloseByClickType(@Nullable GdRfCloseByClick gdRfCloseByClick) {
        if (gdRfCloseByClick == null) {
            return null;
        }

        return TO_RF_CLOSE_BY_CLICK_TYPE_MAP.get(gdRfCloseByClick);
    }

    public static GdImpressionStandardTime toGdImpressionStandardTime(ImpressionStandardTime impressionStandardTime) {
        return ifNotNull(impressionStandardTime, time -> GdImpressionStandardTime.valueOf(time.name().toUpperCase()));
    }

    public static GdEshowsRate toGdEshowsRate(EshowsRate eshowsRate) {
        return ifNotNull(eshowsRate, type -> GdEshowsRate.valueOf(type.name().toUpperCase()));
    }

    public static GdEshowsVideoType toGdEshowsVideoType(EshowsVideoType videoType) {
        return ifNotNull(videoType, type -> GdEshowsVideoType.valueOf(type.name().toUpperCase()));
    }

    private static GdEshowsSettings toGdEshowsSettings(@Nullable EshowsSettings eshowsSettings) {
        if (eshowsSettings == null) {
            return new GdEshowsSettings();
        }

        return new GdEshowsSettings()
                .withBannerRate(toGdEshowsRate(eshowsSettings.getBannerRate()))
                .withVideoRate(toGdEshowsRate(eshowsSettings.getVideoRate()))
                .withVideoType(toGdEshowsVideoType(eshowsSettings.getVideoType()));
    }

    private static GdDayBudgetShowMode toGdDayBudgetShowMode(GdiDayBudgetShowMode dayBudgetShowMode) {
        return GdDayBudgetShowMode.valueOf(dayBudgetShowMode.name());
    }

    public static GdDayBudgetShowMode toGdDayBudgetShowMode(DayBudgetShowMode dayBudgetShowMode) {
        return GdDayBudgetShowMode.valueOf(dayBudgetShowMode.name());
    }

    public static Set<GdCampaignPlacementType> toGdPlacementTypes(Set<PlacementType> placementTypes) {
        return mapSet(placementTypes, placementType -> GdCampaignPlacementType.valueOf(placementType.name()));
    }

    @Nullable
    static List<String> toAllowedDomains(@Nullable List<String> allowedDomains, @Nullable List<String> allowedSsp) {
        if (allowedSsp == null && allowedDomains == null) {
            return null;
        }
        List<String> list = new ArrayList<>();
        if (allowedSsp != null) {//ssp выше в итоговом списке
            list.addAll(allowedSsp);
        }
        if (allowedDomains != null) {
            list.addAll(allowedDomains);
        }
        return list;
    }

    @Nullable
    static Set<String> toDisabledPlacesSet(@Nullable String disabledDomains, @Nullable String disabledSsp) {
        Set<String> domains = toDisabledDomainsSet(disabledDomains);
        Set<String> ssp = toDisabledSspSet(disabledSsp);

        Set<String> places = Sets.union(domains, ssp);

        return places.isEmpty() ? null : places;
    }

    private static Set<String> toDisabledDomainsSet(@Nullable String coreDisabledDomains) {
        return nvl(ifNotNull(coreDisabledDomains,
                disabledDomains -> StreamEx.
                        split(disabledDomains, DISABLED_DOMAINS_DELIMITER)
                        .filter(s -> !s.isBlank())
                        .map(String::strip)
                        .toSet()
        ), emptySet());
    }

    private static Set<String> toDisabledSspSet(@Nullable String coreDisabledSsp) {
        //noinspection unchecked
        return nvl(ifNotNull(coreDisabledSsp,
                disabledSsp -> StreamEx.of((Set<String>) fromJson(disabledSsp, Set.class))
                        .filter(s -> !s.isBlank())
                        .map(String::strip)
                        .toSet()
        ), emptySet());
    }

    @Nullable
    private static List<String> toDisabledIps(@Nullable String coreDisabledIps) {
        if (StringUtils.isEmpty(coreDisabledIps)) {
            return null;
        }
        return StreamEx.split(coreDisabledIps, DISABLED_IPS_DELIMITER).toList();
    }

    @Nullable
    public static ContentLanguage toContentLanguage(@Nullable GdContentLanguage contentLanguage) {
        return ifNotNull(contentLanguage, cl -> ContentLanguage.valueOf(cl.name().toUpperCase()));
    }

    @Nullable
    public static CampaignAttributionModel toCampaignAttributionModel(@Nullable GdCampaignAttributionModel attributionModel) {
        return ifNotNull(attributionModel, am -> CampaignAttributionModel.valueOf(am.name().toUpperCase()));
    }

    @Nullable
    public static List<String> getPromocodeRestrictedDomainViews(@Nullable String domain) {
        if (domain == null) {
            return null;
        }
        List<String> domainViews = new ArrayList<>();
        domainViews.add(domain);
        if (!domain.startsWith(WWW)) {
            domainViews.add(WWW + "." + domain);
        }
        if (domain.contains("рф")) {
            List<String> punyCodes = mapList(domainViews, IDN::toASCII);
            domainViews.addAll(punyCodes);
        }
        return domainViews;
    }

    @Nullable
    private static GdContentLanguage toGdContentLanguage(@Nullable GdiContentLanguage gdiContentLanguage) {
        return ifNotNull(gdiContentLanguage, gcl -> GdContentLanguage.valueOf(gcl.name().toUpperCase()));
    }

    public static GdTimeTarget toGdTimeTarget(TimeTarget timeTarget) {
        GdTimeTarget result = new GdTimeTarget();
        List<List<Integer>> timeBoard = new ArrayList<>();
        for (WeekdayType type : WeekdayType.values()) {
            HoursCoef hoursCoef = timeTarget.getWeekdayCoefs().get(type);

            if (type == WeekdayType.HOLIDAY) {
                result.setEnabledHolidaysMode(timeTarget.getWeekdayCoefs().containsKey(type));
                GdHolidaysSettings settings = getHolidaysSettings(hoursCoef);
                result.setHolidaysSettings(settings);
            } else if (type == WeekdayType.WORKING_WEEKEND) {
                result.setUseWorkingWeekends(timeTarget.getWeekdayCoefs().containsKey(type));
            } else {
                if (hoursCoef == null) { // No coefs, let's fill this day with zeros
                    timeBoard.add(Collections.nCopies(HOURS_PER_DAY, 0));
                } else {
                    timeBoard.add(calculateHourCoefs(hoursCoef));
                }
            }
        }

        return result.withTimeBoard(timeBoard);
    }

    public static TimeTarget toTimeTarget(GdTimeTarget timeTarget) {
        TimeTarget result = new TimeTarget();

        // Collect regular ordinary days:
        for (int i = 0; i < DAYS_PER_WEEK; i++) {
            WeekdayType day = WeekdayType.getById(i + 1);
            List<Integer> coefsByDay = timeTarget.getTimeBoard().get(i);
            HoursCoef hoursCoef = new HoursCoef();

            for (int j = 0; j < HOURS_PER_DAY; j++) {
                hoursCoef.setCoef(j, coefsByDay.get(j));
            }

            result.setWeekdayCoef(day, hoursCoef);
        }

        if (timeTarget.getEnabledHolidaysMode()) {
            HoursCoef hoursCoef = new HoursCoef();
            result.setWeekdayCoef(WeekdayType.HOLIDAY, hoursCoef);
            if (timeTarget.getHolidaysSettings().getIsShow()) {
                GdHolidaysSettings settings = timeTarget.getHolidaysSettings();
                for (int i = settings.getStartHour(); i < settings.getEndHour(); i++) {
                    int coef = settings.getRateCorrections() == null ?
                            TimeTarget.PredefinedCoefs.USUAL.getValue() : settings.getRateCorrections();
                    hoursCoef.setCoef(i, coef);
                }
            }
        }

        if (timeTarget.getUseWorkingWeekends()) {
            result.setWeekdayCoef(WeekdayType.WORKING_WEEKEND, new HoursCoef());
        }

        return result;
    }

    private static List<Integer> calculateHourCoefs(HoursCoef hoursCoef) {
        return IntStreamEx.range(0, HOURS_PER_DAY)
                .boxed()
                .map(hoursCoef::getCoefForHour)
                .toList();
    }

    private static GdHolidaysSettings getHolidaysSettings(@Nullable HoursCoef coefs) {
        GdHolidaysSettings settings = new GdHolidaysSettings();
        if (coefs == null) {
            settings.withIsShow(false);
            return settings;
        }

        Integer startHour = null;
        Integer endHour = null;
        Integer coef = null;
        for (int i = 0; i < HOURS_PER_DAY; i++) {
            if (coefs.getCoefForHour(i) > 0) {
                if (coef == null || coef == 0) {
                    coef = coefs.getCoefForHour(i);
                }
                if (startHour == null) {
                    startHour = i;
                }
                endHour = i + 1;
            }
        }
        settings
                .withIsShow(startHour != null) // In this case endHour is not null too
                .withStartHour(startHour)
                .withEndHour(endHour)
                .withRateCorrections(coef);
        return settings;
    }

    private static GdBroadMatch toGdBroadMatch(GdiCampaign gdiCampaign) {
        return new GdBroadMatch()
                .withBroadMatchFlag(gdiCampaign.getHasBroadMatch())
                .withBroadMatchLimit(gdiCampaign.getBroadMatchLimit())
                .withBroadMatchGoalId(gdiCampaign.getBroadMatchGoalId());
    }

    public static GdCampaignAttributionModel toGdAttributionModel(
            CampaignAttributionModel campaignsAttributionModel) {
        return GdCampaignAttributionModel.valueOf(campaignsAttributionModel.name().toUpperCase());
    }

    private static Set<GdCpmYndxFrontpageCampaignShowType> toGdCpmYndxFrontpageCampaignShowType(
            Set<FrontpageCampaignShowType> frontpageCampaignShowTypes) {
        return mapSet(frontpageCampaignShowTypes,
                campaignShowType -> GdCpmYndxFrontpageCampaignShowType.valueOf(campaignShowType.name().toUpperCase()));
    }

    private static Set<GdMobileContentCampaignDeviceTypeTargeting> toGdMobileContentCampaignDeviceTypeTargeting(
            Set<MobileAppDeviceTypeTargeting> deviceTypeTargetings) {
        return mapSet(deviceTypeTargetings,
                deviceType -> GdMobileContentCampaignDeviceTypeTargeting.valueOf(deviceType.name().toUpperCase()));
    }

    private static Set<GdMobileContentCampaignNetworkTargeting> toGdMobileContentCampaignNetworkTargeting(
            Set<MobileAppNetworkTargeting> networkTargetings) {
        return mapSet(networkTargetings,
                network -> GdMobileContentCampaignNetworkTargeting.valueOf(network.name().toUpperCase()));
    }

    public static GdMobileContentInfo toGdMobileContentInfo(MobileContentInfo mobileContentInfo) {
        return new GdMobileContentInfo()
                .withOsType(GdMobileContentCampaignOsType.valueOf(mobileContentInfo.getOsType().name()))
                .withStoreContentId(mobileContentInfo.getStoreContentId())
                .withStoreCountry(mobileContentInfo.getStoreCountry())
                .withStoreHref(mobileContentInfo.getStoreHref())
                .withMinimalOsVersionFromStore(mobileContentInfo.getMinimalOsVersionFromStore())
                .withCurrentMinimalOsVersion(mobileContentInfo.getCurrentMinimalOsVersion())
                .withIconHash(mobileContentInfo.getIconHash())
                .withIconUrl(mobileContentInfo.getIconUrl())
                .withMobileContentName(mobileContentInfo.getMobileContentName())
                .withTracker(ifNotNull(mobileContentInfo.getTracker(),
                        CampaignDataConverter::toGdMobileContentTracker));
    }

    private static GdMobileAppTracker toGdMobileContentTracker(MobileAppTracker mobileAppTracker) {
        return new GdMobileAppTracker()
                .withTrackerId(mobileAppTracker.getTrackerId())
                .withTrackingSystem(GdTrackingSystem.valueOf(mobileAppTracker.getTrackingSystem().name()))
                .withUrl(mobileAppTracker.getUrl())
                .withImpressionUrl(mobileAppTracker.getImpressionUrl());
    }

    @Nullable
    public static List<GdMeaningfulGoal> toGdMeaningFulGoals(@Nullable List<MeaningfulGoal> goals) {
        return mapList(goals, goal -> new GdMeaningfulGoal()
                .withGoalId(goal.getGoalId())
                .withConversionValue(goal.getConversionValue())
                .withIsMetrikaSourceOfValue(nvl(goal.getIsMetrikaSourceOfValue(), false))
        );
    }

    /**
     * Возвращает статус сервисируемости кампании
     */
    public static GdCampaignServicedState extractCampaignServicedState(Set<GdCampaignAction> effectivePseudoActions) {
        if (effectivePseudoActions.contains(GdCampaignAction.SERVICED)) {
            return GdCampaignServicedState.SERVICED;
        } else if (effectivePseudoActions.contains(GdCampaignAction.ACCEPT_SERVICING)) {
            return GdCampaignServicedState.ACCEPT_SERVICING;
        }

        return GdCampaignServicedState.SELF_SERVICED;
    }

    public static InternalCampaignRestrictionType toInternalCampaignRestrictionType(
            GdInternalFreeCampaignRestrictionType restrictionType) {
        return InternalCampaignRestrictionType.valueOf(restrictionType.name());
    }

    public static GdCampaignMeasurer toGdCampaignMeasurer(CampaignMeasurer core) {
        return new GdCampaignMeasurer()
                .withMeasurerSystem(toGdMeasurerSystem(core.getMeasurerSystem()))
                .withParams(toGdMeasureParams(core));
    }

    private static String toGdMeasureParams(CampaignMeasurer core) {
        switch (core.getMeasurerSystem()) {
            case MOAT:
            case DV:
                return core.getParams();
            case IAS:
                var params = fromJson(core.getParams(), CampaignMeasurerParamsIas.class);
                return params.toGd();

            default:
                return null;
        }
    }

    private static boolean getUseCurrentRegion(@NotNull Boolean useCurrentRegion,
                                               @NotNull Boolean useRegularRegion,
                                               boolean advancedGeoTargeting) {
        if (!advancedGeoTargeting) {
            return false;
        }
        return useCurrentRegion || !useRegularRegion;
    }

    private static boolean getUseRegularRegion(@NotNull Boolean useCurrentRegion,
                                               @NotNull Boolean useRegularRegion,
                                               @NotNull Boolean hasExtendedGeotargeting,
                                               boolean advancedGeoTargeting) {
        if (!advancedGeoTargeting) {
            return false;
        }
        return useRegularRegion || !useCurrentRegion && hasExtendedGeotargeting;
    }

    public static GdCampaignMeasurerSystem toGdMeasurerSystem(CampaignMeasurerSystem core) {
        switch (core) {
            case MOAT:
                return GdCampaignMeasurerSystem.MOAT;
            case DV:
                return GdCampaignMeasurerSystem.DV;
            case IAS:
                return GdCampaignMeasurerSystem.IAS;
            default:
                return null;
        }
    }

    public static GdMetaCampaignFilterType toMetaCampaignFilterType(GdiCampaign campaign) {
        if (campaign.getSource() == GdiCampaignSource.UAC) {
            switch (campaign.getType()) {
                case TEXT:
                    if (campaign.getMetatype() == GdiCampaignMetatype.ECOM) {
                        return GdMetaCampaignFilterType.UC_ECOM;
                    } else {
                        return GdMetaCampaignFilterType.UC_TEXT;
                    }
                case CPM_BANNER:
                    return GdMetaCampaignFilterType.UC_CPM_BANNER;
                case MOBILE_CONTENT:
                    return GdMetaCampaignFilterType.UC_MOBILE_CONTENT;
            }
        }
        return GdMetaCampaignFilterType.valueOf(campaign.getType().name());
    }

    @Nullable
    public static Long getEffectiveStrategyId(GdiCampaign gdiCampaign) {
        var strategyId = gdiCampaign.getStrategyId();
        return strategyId == null || strategyId == 0L ? null : strategyId;
    }

}
