package ru.yandex.direct.grid.model.entity.campaign.converter;

import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;

import ru.yandex.direct.core.entity.campaign.model.CampaignType;
import ru.yandex.direct.core.entity.campaign.model.CampaignsPlatform;
import ru.yandex.direct.grid.model.campaign.GdCampaignFeature;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaignMediaplanStatus;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;

@ParametersAreNonnullByDefault
public class GdCampaignFeatureHelper {

    private static final Set<CampaignType> CPM_TYPES =
            Set.of(CampaignType.CPM_DEALS, CampaignType.CPM_BANNER,
                    CampaignType.CPM_PRICE, CampaignType.CPM_YNDX_FRONTPAGE);
    private static final Set<CampaignType> TYPES_SUPPORT_AIMING_ON_SEARCH =
            Set.of(CampaignType.TEXT, CampaignType.DYNAMIC);

    public static Set<GdCampaignFeature> getCampaignFeatures(GdiCampaign campaign, GdCampaignFlatStrategy strategy) {
        FeatureCheckNode node = new FeatureCheckNode(campaign, strategy);
        return EntryStream.of(FEATURE_PREDICATES)
                .mapValues(p -> p.test(node))
                .filterValues(Boolean.TRUE::equals)
                .keys()
                .toSet();
    }

    private final static Map<GdCampaignFeature, Predicate<FeatureCheckNode>> FEATURE_PREDICATES = Map.ofEntries(
            Map.entry(GdCampaignFeature.SHOW_GENERAL_PRICE_ON_GROUP_EDIT, n -> showGroupGeneralPriceOnEdit(n.strategy)),
            Map.entry(GdCampaignFeature.IS_AIMING_ALLOWED, n -> isCampaignAimingAllowed(n.campaign)),
            Map.entry(GdCampaignFeature.HAS_ADS, n -> n.campaign.getHasBanners()),
            Map.entry(GdCampaignFeature.HAS_NOT_ARCHIVED_ADS, n -> n.campaign.getHasNotArchiveBanners()),
            Map.entry(GdCampaignFeature.HAS_ACTIVE_ADS, n -> n.campaign.getHasActiveBanners()),
            Map.entry(GdCampaignFeature.HAS_BID_MODIFIERS, n -> n.campaign.getHasBidModifiers()),
            Map.entry(GdCampaignFeature.HAS_COMPLETED_MEDIAPLAN,
                    n -> GdCampaignFeatureHelper.hasCompletedMediaplan(n.campaign))
    );

    public static boolean hasCompletedMediaplan(GdiBaseCampaign campaign) {
        return campaign.getMediaplanStatus() == GdiCampaignMediaplanStatus.COMPLETE
                && campaign.getHasNewMediaplan() && campaign.getHasMediaplanBanners();
    }

    static boolean isCampaignAimingAllowed(GdiCampaign campaign) {
        return CPM_TYPES.contains(campaign.getType()) || (TYPES_SUPPORT_AIMING_ON_SEARCH.contains(campaign.getType())
                && campaign.getPlatform() != CampaignsPlatform.SEARCH);
    }

    private static boolean showGroupGeneralPriceOnEdit(GdCampaignFlatStrategy campaignFlatStrategy) {
        return !campaignFlatStrategy.getIsAutoBudget();
    }

    private static class FeatureCheckNode {
        private final GdiCampaign campaign;
        private final GdCampaignFlatStrategy strategy;

        public FeatureCheckNode(GdiCampaign campaign, GdCampaignFlatStrategy strategy) {
            this.campaign = campaign;
            this.strategy = strategy;
        }
    }

}

