package ru.yandex.direct.grid.model.entity.campaign.converter;

import java.util.List;
import java.util.Map;
import java.util.function.UnaryOperator;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.BiMap;
import com.google.common.collect.ImmutableBiMap;
import com.google.common.collect.ImmutableMap;

import ru.yandex.direct.core.entity.campaign.model.PriceFlightReasonIncorrect;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusApprove;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightStatusCorrect;
import ru.yandex.direct.core.entity.campaign.model.PriceFlightTargetingsSnapshot;
import ru.yandex.direct.core.entity.pricepackage.model.ViewType;
import ru.yandex.direct.grid.model.campaign.GdPriceFlightReasonIncorrect;
import ru.yandex.direct.grid.model.campaign.GdPriceFlightStatusApprove;
import ru.yandex.direct.grid.model.campaign.GdPriceFlightStatusCorrect;
import ru.yandex.direct.grid.model.campaign.GdPriceFlightTargetingsSnapshot;
import ru.yandex.direct.grid.model.campaign.GdPriceFlightViewType;

import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@ParametersAreNonnullByDefault
public class PriceFlightDataConverter {

    private static final BiMap<ViewType, GdPriceFlightViewType> TO_GD_PRICE_FLIGHT_VIEW_TYPE_MAP =
            ImmutableBiMap.<ViewType, GdPriceFlightViewType>builder()
                    .put(ViewType.MOBILE, GdPriceFlightViewType.MOBILE)
                    .put(ViewType.DESKTOP, GdPriceFlightViewType.DESKTOP)
                    .put(ViewType.NEW_TAB, GdPriceFlightViewType.NEW_TAB)
                    .build();
    private static final Map<GdPriceFlightViewType, ViewType> TO_PRICE_FLIGHT_VIEW_TYPE_MAP =
            TO_GD_PRICE_FLIGHT_VIEW_TYPE_MAP.inverse();

    private static final Map<PriceFlightStatusApprove, GdPriceFlightStatusApprove>
            TO_GD_PRICE_FLIGHT_STATUS_APPROVE_MAP =
            ImmutableMap.<PriceFlightStatusApprove, GdPriceFlightStatusApprove>builder()
                    .put(PriceFlightStatusApprove.NEW, GdPriceFlightStatusApprove.NEW)
                    .put(PriceFlightStatusApprove.NO, GdPriceFlightStatusApprove.NO)
                    .put(PriceFlightStatusApprove.YES, GdPriceFlightStatusApprove.YES)
                    .build();

    private static final Map<PriceFlightStatusCorrect, GdPriceFlightStatusCorrect>
            TO_GD_PRICE_FLIGHT_STATUS_CORRECT_MAP =
            ImmutableMap.<PriceFlightStatusCorrect, GdPriceFlightStatusCorrect>builder()
                    .put(PriceFlightStatusCorrect.NEW, GdPriceFlightStatusCorrect.NEW)
                    .put(PriceFlightStatusCorrect.NO, GdPriceFlightStatusCorrect.NO)
                    .put(PriceFlightStatusCorrect.YES, GdPriceFlightStatusCorrect.YES)
                    .build();

    private static final Map<PriceFlightReasonIncorrect, GdPriceFlightReasonIncorrect>
            TO_GD_PRICE_FLIGHT_REASON_INCORRECT_MAP =
            ImmutableMap.<PriceFlightReasonIncorrect, GdPriceFlightReasonIncorrect>builder()
                    .put(PriceFlightReasonIncorrect.NO_DEFAULT_GROUP,
                            GdPriceFlightReasonIncorrect.NO_DEFAULT_GROUP)
                    .put(PriceFlightReasonIncorrect.MORE_THAN_ONE_DEFAULT_GROUP,
                            GdPriceFlightReasonIncorrect.MORE_THAN_ONE_DEFAULT_GROUP)
                    .put(PriceFlightReasonIncorrect.NOT_MODERATED_DEFAULT_GROUP,
                            GdPriceFlightReasonIncorrect.NOT_MODERATED_DEFAULT_GROUP)
                    .put(PriceFlightReasonIncorrect.NOT_MODERATED_GROUP,
                            GdPriceFlightReasonIncorrect.NOT_MODERATED_GROUP)
                    .put(PriceFlightReasonIncorrect.NO_ACTIVE_BANNERS,
                            GdPriceFlightReasonIncorrect.NO_ACTIVE_BANNERS)
                    .put(PriceFlightReasonIncorrect.NOT_FULL,
                            GdPriceFlightReasonIncorrect.NOT_FULL)
                    .build();

    private PriceFlightDataConverter() {
    }

    public static PriceFlightTargetingsSnapshot toPriceFlightTargetingsSnapshot(
            GdPriceFlightTargetingsSnapshot priceFlightTargetingsSnapshot) {
        return new PriceFlightTargetingsSnapshot()
                .withGeoType(priceFlightTargetingsSnapshot.getGeoType())
                .withGeoExpanded(priceFlightTargetingsSnapshot.getGeoExpanded())
                .withViewTypes(mapList(priceFlightTargetingsSnapshot.getViewTypes(),
                        TO_PRICE_FLIGHT_VIEW_TYPE_MAP::get))
                .withAllowExpandedDesktopCreative(priceFlightTargetingsSnapshot.getAllowExpandedDesktopCreative());
    }

    public static GdPriceFlightTargetingsSnapshot toGdPriceFlightTargetingsSnapshot(
            PriceFlightTargetingsSnapshot priceFlightTargetingsSnapshot,
            UnaryOperator<List<Long>> geoToFrontendGeo) {
        return new GdPriceFlightTargetingsSnapshot()
                .withGeoType(priceFlightTargetingsSnapshot.getGeoType())
                .withGeoExpanded(geoToFrontendGeo.apply(priceFlightTargetingsSnapshot.getGeoExpanded()))
                .withViewTypes(mapList(priceFlightTargetingsSnapshot.getViewTypes(),
                        TO_GD_PRICE_FLIGHT_VIEW_TYPE_MAP::get))
                .withAllowExpandedDesktopCreative(priceFlightTargetingsSnapshot.getAllowExpandedDesktopCreative());
    }

    public static GdPriceFlightStatusApprove toGdPriceFlightStatusApprove(PriceFlightStatusApprove statusApprove) {
        return TO_GD_PRICE_FLIGHT_STATUS_APPROVE_MAP.get(statusApprove);
    }

    public static GdPriceFlightStatusCorrect toGdPriceFlightStatusCorrect(PriceFlightStatusCorrect statusCorrect) {
        return TO_GD_PRICE_FLIGHT_STATUS_CORRECT_MAP.get(statusCorrect);
    }

    @Nullable
    public static GdPriceFlightReasonIncorrect toGdPriceFlightReasonIncorrect(
            @Nullable PriceFlightReasonIncorrect reasonIncorrect) {
        return reasonIncorrect == null ? null
                : TO_GD_PRICE_FLIGHT_REASON_INCORRECT_MAP.get(reasonIncorrect);
    }
}
