package ru.yandex.direct.grid.model.entity.campaign.strategy;

import java.time.LocalDate;
import java.util.Map;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.collect.ImmutableSet;

import ru.yandex.direct.grid.model.campaign.GdCampaignType;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyGroup;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignFlatStrategy;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyType;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType;

import static com.google.common.base.Preconditions.checkArgument;
import static com.google.common.base.Preconditions.checkNotNull;
import static ru.yandex.direct.utils.CommonUtils.nvl;

@ParametersAreNonnullByDefault
public class GdStrategyExtractorFacade {
    private final Map<GdStrategyType, GdStrategyExtractor<?>> supportMap;

    /**
     * Типы кампаний, которые могут показываться и на поиске, и в сетях
     */
    private static final Set<GdCampaignType> CAMPAIGN_TYPES_WITH_SEPARATE_BIDDING_SUPPORT = ImmutableSet.of(
            GdCampaignType.TEXT,
            GdCampaignType.MOBILE_CONTENT
    );


    public GdStrategyExtractorFacade(Map<GdStrategyType, GdStrategyExtractor<?>> supportMap) {
        this.supportMap = supportMap;
    }

    @Deprecated
    public GdCampaignFlatStrategy extractFlatStrategy(GdiCampaignStrategyGroup campaignStrategyGroup) {
        GdCampaignStrategyType strategyType = GdStrategyExtractorHelper.toCampaignStrategyType(campaignStrategyGroup);
        return getExtractorForType(strategyType).extractFlatStrategy(campaignStrategyGroup, strategyType);
    }

    public GdCampaignFlatStrategy extractStrategy(GdiCampaignStrategyGroup campaignStrategyGroup) {
        GdStrategyType strategyType = GdStrategyExtractorHelper.toStrategyType(campaignStrategyGroup);
        return getExtractorForType(strategyType).extractStrategy(campaignStrategyGroup, strategyType);
    }

    public boolean hasSeparateBiddingSupport(GdCampaignFlatStrategy gdCampaignFlatStrategy,
                                             GdCampaignType campaignType) {
        return getExtractorForType(gdCampaignFlatStrategy.getType()).hasSeparateBiddingSupport() &&
                CAMPAIGN_TYPES_WITH_SEPARATE_BIDDING_SUPPORT.contains(campaignType);
    }

    public boolean isAutoBudget(GdiCampaignStrategyGroup campaignStrategyGroup) {
        GdStrategyType strategyType = GdStrategyExtractorHelper.toStrategyType(campaignStrategyGroup);
        return getExtractorForType(strategyType).isTypeAutoBudget();
    }

    public boolean isPayForConversionEnabled(GdCampaignFlatStrategy gdCampaignFlatStrategy) {
        return nvl(getExtractorForType(gdCampaignFlatStrategy.getStrategyType()).isPayForConversionEnabled(gdCampaignFlatStrategy), false);
    }

    @Nullable
    public LocalDate extractStrategyLastRestartTime(GdCampaignFlatStrategy gdCampaignFlatStrategy) {
        return getExtractorForType(gdCampaignFlatStrategy.getStrategyType()).getLastBidderRestartTime(gdCampaignFlatStrategy);
    }

    @Nullable
    public Long extractStrategyGoalId(GdCampaignFlatStrategy gdCampaignFlatStrategy) {
        return getExtractorForType(gdCampaignFlatStrategy.getStrategyType()).getGoalId(gdCampaignFlatStrategy);
    }

    public Long extractStrategyGoalIdAndVerifyNonNull(GdCampaignFlatStrategy gdCampaignFlatStrategy) {
        var goalId = getExtractorForType(gdCampaignFlatStrategy.getStrategyType()).getGoalId(gdCampaignFlatStrategy);
        checkNotNull(goalId, "goalId must be set");
        return goalId;
    }

    private GdStrategyExtractor getExtractorForType(GdStrategyType type) {
        checkArgument(supportMap.containsKey(type), "This strategy type: " + type + " is not supported");
        return supportMap.get(type);
    }

    @Deprecated
    private GdStrategyExtractor getExtractorForType(GdCampaignStrategyType campaignStrategyType) {
        GdStrategyType strategyType = GdStrategyType.valueOf(campaignStrategyType.name());
        checkArgument(supportMap.containsKey(strategyType), "This strategy type: " + strategyType + " is not " +
                "supported");
        return supportMap.get(strategyType);
    }

}
