package ru.yandex.direct.grid.model.entity.campaign.strategy;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Optional;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import ru.yandex.direct.core.entity.campaign.model.StrategyData;
import ru.yandex.direct.core.entity.campaign.repository.CampaignMappings;
import ru.yandex.direct.core.entity.strategy.StrategyExtractorHelper;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyGroup;
import ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName;
import ru.yandex.direct.grid.model.campaign.GdiDayBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudget;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetPeriod;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignBudgetShowMode;
import ru.yandex.direct.grid.model.campaign.strategy.GdCampaignStrategyType;
import ru.yandex.direct.grid.model.campaign.strategy.GdStrategyType;

import static com.google.common.base.Preconditions.checkArgument;
import static java.util.Map.entry;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CLICK;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPA;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_CAMP;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPA_PER_FILTER;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPC_PER_CAMP;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPC_PER_FILTER;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPI;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPV;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_CRR;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_MAX_IMPRESSIONS;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_MAX_REACH;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_ROI;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.AUTOBUDGET_WEEK_BUNDLE;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.CPM_DEFAULT;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.DEFAULT_;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.NO_PREMIUM;
import static ru.yandex.direct.grid.model.campaign.GdiCampaignStrategyName.PERIOD_FIX_BID;
import static ru.yandex.direct.utils.CommonUtils.ifNotNull;

@ParametersAreNonnullByDefault
public class GdStrategyExtractorHelper {
    public static final Map<GdStrategyType, GdStrategyExtractor<?>> STRATEGIES_EXTRACTORS_BY_TYPES = Map.ofEntries(
            entry(GdStrategyType.DEFAULT, new GdStrategyManualExtractor()),
            entry(GdStrategyType.NO_PREMIUM, new GdStrategyNoPremiumExtractor()),
            entry(GdStrategyType.WEEK_BUDGET, new GdStrategyWeekBudgetExtractor()),
            entry(GdStrategyType.WEEK_BUNDLE, new GdStrategyWeekBundleExtractor()),
            entry(GdStrategyType.AVG_CLICK, new GdStrategyAvgClickExtractor()),
            entry(GdStrategyType.AVG_CPA, new GdStrategyAvgCpaExtractor()),
            entry(GdStrategyType.AVG_CPA_PER_CAMP, new GdStrategyAvgCpaPerCampExtractor()),
            entry(GdStrategyType.AVG_CPA_PER_FILTER, new GdStrategyAvgCpaPerFilterExtractor()),
            entry(GdStrategyType.AVG_CPC_PER_CAMP, new GdStrategyAvgCpcPerCampExtractor()),
            entry(GdStrategyType.AVG_CPC_PER_FILTER, new GdStrategyAvgCpcPerFilterExtractor()),
            entry(GdStrategyType.AVG_CPI, new GdStrategyAvgCpiExtractor()),
            entry(GdStrategyType.ROI, new GdStrategyRoiExtractor()),
            entry(GdStrategyType.CRR, new GdStrategyCrrExtractor()),
            entry(GdStrategyType.CPM_DEFAULT, new GdStrategyCpmDefaultExtractor()),
            entry(GdStrategyType.CPM_MAX_REACH, new GdStrategyCpmMaxReachExtractor()),
            entry(GdStrategyType.CPM_MAX_IMPRESSIONS, new GdStrategyCpmMaxImpressionsExtractor()),
            entry(GdStrategyType.CPM_MAX_REACH_CUSTOM_PERIOD, new GdStrategyCpmMaxReachCustomPeriodExtractor()),
            entry(GdStrategyType.CPM_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                    new GdStrategyCpmMaxImpressionsCustomPeriodExtractor()),
            entry(GdStrategyType.AVG_CPV, new GdStrategyAvgCpvExtractor()),
            entry(GdStrategyType.AVG_CPV_CUSTOM_PERIOD, new GdStrategyAvgCpvCustomPeriodExtractor()),
            entry(GdStrategyType.OPTIMIZE_CLICKS, new GdStrategyOptimizeClicksExtractor()),
            entry(GdStrategyType.OPTIMIZE_CONVERSIONS, new GdStrategyOptimizeConversionsExtractor()),
            entry(GdStrategyType.OPTIMIZE_INSTALLS, new GdStrategyOptimizeInstallsExtractor()),
            entry(GdStrategyType.PERIOD_FIX_BID, new GdStrategyPeriodFixBidExtractor())
    );

    private static final Map<GdiCampaignStrategyName, GdCampaignStrategyType> GD_OLD_STRATEGY_TYPE_BY_STRATEGY_NAME =
            Map.ofEntries(
                    entry(DEFAULT_, GdCampaignStrategyType.DEFAULT),
                    entry(NO_PREMIUM, GdCampaignStrategyType.NO_PREMIUM),
                    entry(AUTOBUDGET, GdCampaignStrategyType.WEEK_BUDGET),
                    entry(AUTOBUDGET_WEEK_BUNDLE, GdCampaignStrategyType.WEEK_BUNDLE),
                    entry(AUTOBUDGET_AVG_CLICK, GdCampaignStrategyType.AVG_CLICK),
                    entry(AUTOBUDGET_AVG_CPA, GdCampaignStrategyType.AVG_CPA),
                    entry(AUTOBUDGET_AVG_CPA_PER_CAMP, GdCampaignStrategyType.AVG_CPA_PER_CAMP),
                    entry(AUTOBUDGET_AVG_CPA_PER_FILTER, GdCampaignStrategyType.AVG_CPA_PER_FILTER),
                    entry(AUTOBUDGET_AVG_CPC_PER_CAMP, GdCampaignStrategyType.AVG_CPC_PER_CAMP),
                    entry(AUTOBUDGET_AVG_CPC_PER_FILTER, GdCampaignStrategyType.AVG_CPC_PER_FILTER),
                    entry(AUTOBUDGET_AVG_CPI, GdCampaignStrategyType.AVG_CPI),
                    entry(AUTOBUDGET_ROI, GdCampaignStrategyType.ROI),
                    entry(AUTOBUDGET_CRR, GdCampaignStrategyType.CRR),
                    entry(CPM_DEFAULT, GdCampaignStrategyType.CPM_DEFAULT),
                    entry(AUTOBUDGET_MAX_REACH, GdCampaignStrategyType.CPM_MAX_REACH),
                    entry(AUTOBUDGET_MAX_REACH_CUSTOM_PERIOD,
                            GdCampaignStrategyType.CPM_MAX_REACH_CUSTOM_PERIOD),
                    entry(AUTOBUDGET_MAX_IMPRESSIONS, GdCampaignStrategyType.CPM_MAX_IMPRESSIONS),
                    entry(AUTOBUDGET_MAX_IMPRESSIONS_CUSTOM_PERIOD,
                            GdCampaignStrategyType.CPM_MAX_IMPRESSIONS_CUSTOM_PERIOD),
                    entry(AUTOBUDGET_AVG_CPV, GdCampaignStrategyType.AVG_CPV),
                    entry(AUTOBUDGET_AVG_CPV_CUSTOM_PERIOD, GdCampaignStrategyType.AVG_CPV_CUSTOM_PERIOD),
                    entry(PERIOD_FIX_BID, GdCampaignStrategyType.PERIOD_FIX_BID)
            );

    public static GdStrategyType toStrategyType(GdiCampaignStrategyGroup campaignStrategyGroup) {
        var strategyName = GdiCampaignStrategyName.toSource(campaignStrategyGroup.getStrategyName());
        var strategyData = ifNotNull(campaignStrategyGroup.getStrategyData(), CampaignMappings::strategyDataFromDb);
        var campaignType = campaignStrategyGroup.getType();

        var strategyType = StrategyExtractorHelper.toStrategyType(strategyName, strategyData, campaignType);
        return GdStrategyType.fromSource(strategyType);
    }

    @Deprecated
    static GdCampaignStrategyType toCampaignStrategyType(GdiCampaignStrategyGroup campaignStrategyGroup) {
        GdiCampaignStrategyName strategyName = campaignStrategyGroup.getStrategyName();
        checkArgument(GD_OLD_STRATEGY_TYPE_BY_STRATEGY_NAME.containsKey(strategyName),
                "Got campaignStrategyGroup with unsupported strategy " + strategyName);
        return GD_OLD_STRATEGY_TYPE_BY_STRATEGY_NAME.get(strategyName);
    }

    static GdCampaignBudget calcBudgetForStrategiesWithCustomPeriod(@Nullable StrategyData strategyData) {
        return new GdCampaignBudget()
                .withSum(ifNotNull(strategyData, StrategyData::getBudget))
                .withShowMode(GdCampaignBudgetShowMode.DEFAULT)
                .withPeriod(GdCampaignBudgetPeriod.CUSTOM)
                .withStart(ifNotNull(strategyData, StrategyData::getStart))
                .withFinish(ifNotNull(strategyData, StrategyData::getFinish))
                .withAutoProlongation(ifNotNull(strategyData, data -> data.getAutoProlongation() > 0));
    }

    static GdCampaignBudget calcBudgetForStrategiesWithDailyBudgetOnly(GdiCampaignStrategyGroup campaignStrategyGroup) {
        return new GdCampaignBudget()
                .withSum(campaignStrategyGroup.getDayBudget())
                .withPeriod(GdCampaignBudgetPeriod.DAY)
                .withShowMode(campaignStrategyGroup.getDayBudgetShowMode() == GdiDayBudgetShowMode.DEFAULT_ ?
                        GdCampaignBudgetShowMode.DEFAULT : GdCampaignBudgetShowMode.STRETCHED);
    }

    @Nullable
    public static GdCampaignBudget calcBudgetForStrategiesWithWeeklyBudgetOnly(@Nullable StrategyData strategyData) {
        var weeklyBudgetSum = Optional.ofNullable(strategyData)
                .map(StrategyData::getSum)
                .orElse(BigDecimal.ZERO);

        return weeklyBudget(weeklyBudgetSum);
    }

    private static GdCampaignBudget weeklyBudget(BigDecimal budget) {
        return new GdCampaignBudget()
                .withSum(budget)
                .withPeriod(GdCampaignBudgetPeriod.WEEK)
                .withShowMode(GdCampaignBudgetShowMode.DEFAULT);
    }

}
