package ru.yandex.direct.grid.model.mapper;

import java.lang.reflect.AnnotatedElement;
import java.lang.reflect.AnnotatedType;
import java.util.Arrays;

import graphql.schema.GraphQLEnumType;
import graphql.schema.GraphQLEnumValueDefinition;
import io.leangen.graphql.generator.BuildContext;
import io.leangen.graphql.generator.JavaDeprecationMappingConfig;
import io.leangen.graphql.generator.OperationMapper;
import io.leangen.graphql.generator.mapping.TypeMappingEnvironment;
import io.leangen.graphql.generator.mapping.common.EnumMapper;
import io.leangen.graphql.metadata.messages.MessageBundle;
import io.leangen.graphql.util.ClassUtils;

import static graphql.schema.GraphQLEnumType.newEnum;

/**
 * Кастомизирующий маппер для Enum'ов. После обновления spqr 0.9.7->0.9.9 немного ихменился механизм генерации схемы.
 * Был исправлен баг, из-за которого при генерации схемы Enum'ы могли регистрироваться в схеме дважды(Enum и EnumInput).
 * Так как фронт работает со схемой которая содержит этот баг, пришлось добавить этот маппер, который его воспроизводит.
 * Есть тикет DIRECT-91063 в котором постараемся перейти на новую схему без поломки обратной совместимости.
 */
public class InputEnumMapper extends EnumMapper {

    public InputEnumMapper() {
        super(new JavaDeprecationMappingConfig(true, "Deprecated"));
    }

    @Override
    public GraphQLEnumType toGraphQLInputType(String typeName, AnnotatedType javaType, TypeMappingEnvironment env) {
        BuildContext buildContext = env.buildContext;

        GraphQLEnumType.Builder enumBuilder = newEnum()
                .name(buildContext.typeInfoGenerator.generateInputTypeName(javaType, buildContext.messageBundle))
                .description(buildContext.typeInfoGenerator
                        .generateInputTypeDescription(javaType, buildContext.messageBundle));
        buildContext.directiveBuilder.buildEnumTypeDirectives(javaType, buildContext.directiveBuilderParams())
                .forEach(directive ->
                        enumBuilder.withDirective(env.operationMapper.toGraphQLDirective(directive, buildContext)));
        addOptions(enumBuilder, javaType, env.operationMapper, buildContext);
        return enumBuilder.build();
    }

    private void addOptions(GraphQLEnumType.Builder enumBuilder, AnnotatedType javaType,
                            OperationMapper operationMapper, BuildContext buildContext) {
        MessageBundle messageBundle = buildContext.messageBundle;
        Arrays.stream((Enum[]) ClassUtils.getRawType(javaType.getType()).getEnumConstants())
                .map(enumConst -> (Enum<?>) enumConst)
                .forEach(enumConst -> enumBuilder.value(GraphQLEnumValueDefinition.newEnumValueDefinition()
                        .name(getValueName(enumConst, messageBundle))
                        .value(enumConst)
                        .description(getValueDescription(enumConst, messageBundle))
                        .deprecationReason(getValueDeprecationReason(enumConst, messageBundle))
                        .withDirectives(getValueDirectives(enumConst, operationMapper, buildContext))
                        .build()));
    }

    @Override
    public boolean supports(AnnotatedElement element, AnnotatedType type) {
        return ClassUtils.getRawType(type.getType()).isEnum();
    }

    @Override
    protected String getTypeName(AnnotatedType type, BuildContext buildContext) {
        return buildContext.typeInfoGenerator.generateTypeName(type, buildContext.messageBundle);
    }

    @Override
    protected String getInputTypeName(AnnotatedType type, BuildContext buildContext) {
        return buildContext.typeInfoGenerator.generateInputTypeName(type, buildContext.messageBundle);
    }
}
