package ru.yandex.direct.grid.model.mapper

import kotlin.reflect.jvm.kotlinProperty
import kotlin.reflect.jvm.kotlinFunction
import io.leangen.graphql.annotations.GraphQLIgnore
import io.leangen.graphql.annotations.GraphQLNonNull
import io.leangen.graphql.generator.mapping.common.NonNullMapper
import io.leangen.graphql.metadata.TypedElement
import io.leangen.graphql.util.ClassUtils
import org.slf4j.LoggerFactory
import java.lang.reflect.AnnotatedElement
import java.lang.reflect.AnnotatedType
import java.lang.reflect.Field
import java.lang.reflect.Method
import java.lang.reflect.Parameter
import java.util.Optional
import java.util.OptionalDouble
import java.util.OptionalInt
import java.util.OptionalLong
import javax.annotation.Nullable
import kotlin.reflect.KParameter
import kotlin.reflect.KType
import kotlin.reflect.full.hasAnnotation

@GraphQLIgnore
class NonNullMapperCustom: NonNullMapper() {
    companion object {
        private val logger = LoggerFactory.getLogger(NonNullMapperCustom::class.java)
        private val alwaysNullableTypes: Set<Class<*>>  = setOf(
            Optional::class.java,
            OptionalLong::class.java,
            OptionalInt::class.java,
            OptionalDouble::class.java,
        )
    }

    override fun supports(element: AnnotatedElement, type: AnnotatedType): Boolean {
        val result = checkForKotlinNullability(element, type)
        return if (result == KotlinNullabiltyCheckResult.NON_KOTLIN) {
            super.supports(element, type)
        } else {
            result == KotlinNullabiltyCheckResult.NON_NULLABLE
        }
    }

    fun checkForKotlinNullability(element: AnnotatedElement, type: AnnotatedType): KotlinNullabiltyCheckResult {
        val elems = (element as TypedElement).elements
        //в elems 2 элемента: method у методов, field у полей, оба у пропертей, см PublicResolverBuilder
        if (elems.stream().anyMatch { t: AnnotatedElement? -> t != null && t is Field }) { //из-за пропертей сначала смотрим на поля
            val field = elems.stream().filter { t: AnnotatedElement? -> t != null && t is Field }.findFirst().get() as Field
            if (!field.declaringClass.isAnnotationPresent(Metadata::class.java)) {
                return KotlinNullabiltyCheckResult.NON_KOTLIN
            } else {
                return if (isNullableKotlinType(field.kotlinProperty!!.returnType, type)) {
                    KotlinNullabiltyCheckResult.NULLABLE
                } else KotlinNullabiltyCheckResult.NON_NULLABLE
            }
        } else if (elems.stream().anyMatch { t: AnnotatedElement? -> t != null && t is Parameter }) {
            val parameter = elems.stream().filter { t: AnnotatedElement? -> t != null && t is Parameter }.findFirst().get() as Parameter //OperationArgument
            if (!parameter.declaringExecutable.declaringClass.isAnnotationPresent(Metadata::class.java)) {
                return KotlinNullabiltyCheckResult.NON_KOTLIN
            } else {
                val method = parameter.declaringExecutable as Method //Executable-метод или конструктор
                val paramIndex = method.parameters.withIndex()
                    .firstOrNull { indexedValue -> indexedValue.value == parameter } //прямого доступа к индексу у параметра нет
                    ?.index!!
                val kParameter: KParameter = method.kotlinFunction!!.parameters.filter { it.kind == KParameter.Kind.VALUE }[paramIndex]
                return if (isNullableKotlinType(kParameter.type, type)) {
                    KotlinNullabiltyCheckResult.NULLABLE
                } else KotlinNullabiltyCheckResult.NON_NULLABLE
            }
        } else if (elems.stream().anyMatch { t: AnnotatedElement? -> t != null && t is Method }) {
            val method = elems.stream().filter { t: AnnotatedElement? -> t != null && t is Method }.findFirst().get() as Method
            if (!method.declaringClass.isAnnotationPresent(Metadata::class.java)) {
                return KotlinNullabiltyCheckResult.NON_KOTLIN
            } else {
                if (method.kotlinFunction == null) {
                    return KotlinNullabiltyCheckResult.NON_KOTLIN
                }
                return if (isNullableKotlinType(method.kotlinFunction!!.returnType, type)) {
                    KotlinNullabiltyCheckResult.NULLABLE
                } else KotlinNullabiltyCheckResult.NON_NULLABLE
            }
        } else {
            logger.warn("some strange object in graphql schema {}", element.javaType.type.typeName)
            return KotlinNullabiltyCheckResult.NON_KOTLIN
        }
    }

    //Поскольку зовётся в том числе для элементов списков, когда kotlinType соответствует всему списку, делаем так
    private fun isNullableKotlinType(kotlinType: KType, type: AnnotatedType): Boolean {
        if (isOptionalType(type)) {
            return true
        }
        if (type.isAnnotationPresent(GraphQLNonNull::class.java)) {//дописываем внутри ListMapperCustom
            return false
        }
        if (type.isAnnotationPresent(GraphQlNullable::class.java)) {//дописываем внутри ListMapperCustom
            return true
        }
        return kotlinType.isMarkedNullable
    }

    fun isOptionalType(type: AnnotatedType) = alwaysNullableTypes.any { ClassUtils.isSuperClass(it, type) }

    enum class KotlinNullabiltyCheckResult {
        NULLABLE, NON_NULLABLE, NON_KOTLIN
    }
}
