package ru.yandex.direct.grid.processing.configuration;

import java.util.Map;
import java.util.concurrent.ThreadPoolExecutor.CallerRunsPolicy;

import javax.annotation.ParametersAreNonnullByDefault;

import com.google.common.util.concurrent.MoreExecutors;
import one.util.streamex.EntryStream;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.FilterType;
import org.springframework.context.annotation.Import;
import org.springframework.core.task.TaskExecutor;
import org.springframework.scheduling.annotation.EnableAsync;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;

import ru.yandex.direct.common.configuration.FrontDbYdbConfiguration;
import ru.yandex.direct.common.configuration.UacYdbConfiguration;
import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.core.configuration.CoreConfiguration;
import ru.yandex.direct.db.config.DbConfigFactory;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.grid.core.configuration.GridCoreConfiguration;
import ru.yandex.direct.grid.processing.annotations.GridGraphQLService;
import ru.yandex.direct.grid.processing.annotations.PublicGraphQLService;
import ru.yandex.direct.grid.processing.processor.GridGraphQLProcessor;
import ru.yandex.direct.grid.processing.processor.interceptor.GridResolverInterceptor;
import ru.yandex.direct.grid.processing.service.cache.storage.GridCacheStorage;
import ru.yandex.direct.grid.processing.service.cache.storage.LettuceGridCacheStorage;
import ru.yandex.direct.grid.processing.service.dataloader.GridDataLoaderRegistry;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.web.core.configuration.WebCoreConfiguration;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;
import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE_CACHE;

@Configuration
@ComponentScan(
        basePackages = {
                "ru.yandex.direct.grid.processing.service",
                "ru.yandex.direct.grid.processing.processor.interceptor"
        },
        excludeFilters = @ComponentScan.Filter(value = Configuration.class, type = FilterType.ANNOTATION)
)
@Import({
        CoreConfiguration.class,
        GridCoreConfiguration.class,
        WebCoreConfiguration.class,
        FrontDbYdbConfiguration.class,
        UacYdbConfiguration.class,
})
@EnableAsync
@ParametersAreNonnullByDefault
public class GridProcessingConfiguration {
    private static final String TRACING_PREFIX = "grid";
    public static final String GRAPH_QL_PROCESSOR = "gridGraphQLProcessor";
    public static final String PUBLIC_GRAPH_QL_PROCESSOR = "publicGraphQLProcessor";

    private static final int THREAD_POOL_SIZE = 1;
    private static final int QUEUE_CAPACITY = 0;

    @Autowired
    DbConfigFactory dbConfigFactory;

    @Autowired
    private GridResolverInterceptor gridResolverInterceptor;

    @Bean(name = GRAPH_QL_PROCESSOR)
    public GridGraphQLProcessor gridGraphQLProcessor(ApplicationContext applicationContext,
                                                     EnvironmentType environmentType, TraceHelper traceHelper,
                                                     @Qualifier(LETTUCE) LettuceConnectionProvider lettuce,
                                                     GridDataLoaderRegistry dataLoaderRegistry) {
        Map<String, Object> beans = applicationContext.getBeansWithAnnotation(GridGraphQLService.class);
        if (!environmentType.isBeta()) {
            beans = EntryStream.of(beans)
                    .filterValues(
                            bean -> !AopUtils.getTargetClass(bean).getAnnotation(GridGraphQLService.class).isBeta())
                    .toMap();
        }
        return new GridGraphQLProcessor(beans.values(), traceHelper, environmentType, TRACING_PREFIX, lettuce,
                dataLoaderRegistry, gridResolverInterceptor, "grid");
    }

    @Bean(name = PUBLIC_GRAPH_QL_PROCESSOR)
    public GridGraphQLProcessor publicGraphQLProcessor(ApplicationContext applicationContext,
                                                       EnvironmentType environmentType, TraceHelper traceHelper,
                                                       GridDataLoaderRegistry dataLoaderRegistry) {
        Map<String, Object> beans = applicationContext.getBeansWithAnnotation(PublicGraphQLService.class);
        // в публичном graphQL пока не поддерживаем rate limit ограничения, для для которых нужен Lettuce/Redis
        LettuceConnectionProvider noLettuce = null;
        return new GridGraphQLProcessor(beans.values(), traceHelper, environmentType, TRACING_PREFIX, noLettuce,
                dataLoaderRegistry, gridResolverInterceptor, "public-grid");
    }

    @Bean
    public GridCacheStorage gridCacheStorage(
            @Qualifier(LETTUCE_CACHE) LettuceConnectionProvider lettuceCache) {
        return new LettuceGridCacheStorage(lettuceCache);
    }

    /**
     * Service executor для выполнения запроса к геобазе, необходим для того, чтобы обеспечить ограничение времени
     * выполнения запроса. В нем указана {@link CallerRunsPolicy} для того, чтобы таски выполнялись в том же треде из
     * которого были вызваны, но создать executor с 0 тредов нельзя поэтому какие-то таски таки будут попадать на этот
     * {@link TaskExecutor}. Можно было бы воспользоваться {@link MoreExecutors#directExecutor()}, но он не позволяет
     * полностью использовать функционал {@link java.util.concurrent.CompletableFuture}.
     */
    @Bean
    public TaskExecutor geoBaseTaskExecutor() {
        var executor = new ThreadPoolTaskExecutor();
        executor.setDaemon(true);
        executor.setCorePoolSize(THREAD_POOL_SIZE);
        executor.setMaxPoolSize(THREAD_POOL_SIZE);
        executor.setQueueCapacity(QUEUE_CAPACITY);
        executor.setRejectedExecutionHandler(new CallerRunsPolicy());

        return executor;
    }
}
