package ru.yandex.direct.grid.processing.context.container;

import java.time.Instant;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;

import com.google.common.collect.ImmutableList;

import ru.yandex.direct.core.entity.banner.model.BannerWithSystemFields;
import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.grid.core.entity.fetchedfieldresolver.FetchedFieldsResolver;
import ru.yandex.direct.grid.model.campaign.GdiBaseCampaign;
import ru.yandex.direct.grid.model.campaign.GdiCampaign;
import ru.yandex.direct.grid.processing.model.client.GdClientInfo;
import ru.yandex.direct.grid.processing.model.group.GdAdGroup;
import ru.yandex.direct.grid.processing.service.campaign.CampaignInfoService;

import static java.util.Collections.unmodifiableMap;
import static ru.yandex.direct.utils.FunctionalUtils.listToMap;

/**
 * Контекст запроса в ручку graphql
 * может содержать как предварительную информацию о запросе, так и полученные в процессе вычисления данные
 */
public class GridGraphQLContext {

    private final User operator;

    @Nullable
    private final User subjectUser;

    private Instant instant;

    private GdClientInfo queriedClient;
    private FetchedFieldsResolver fetchedFieldsReslover;

    private Map<String, GdClientInfo> clientInfoByLogin;

    @Nullable
    private ImmutableList<GdiCampaign> clientGdiCampaigns;

    @Nullable
    private List<GdiBaseCampaign> clientGdiBaseCampaigns;

    /**
     * Мутабельная мапа для хранения кампаний по их id,
     * может заполняться по мере обработки graphql-запроса
     * в отличие от {@link #getClientGdiCampaigns},
     * в котором хранятся все кампании клиента.
     */
    @Nullable
    private Map<Long, GdiCampaign> gdiCampaignsMap;

    private Map<Long, GdAdGroup> gdAdGroupsMap;
    @Nullable
    private Map<Long, BannerWithSystemFields> mainAdsByAdGroupId;

    private final Set<String> resolversToLogValidationResult;

    public GridGraphQLContext(User operator) {
        this(operator, operator);
    }

    public GridGraphQLContext(User operator, @Nullable User subjectUser) {
        this.operator = operator;
        this.subjectUser = subjectUser;
        this.resolversToLogValidationResult = new HashSet<>();
        this.clientInfoByLogin = new HashMap<>();
    }

    public User getOperator() {
        return operator;
    }

    @Nullable
    public User getSubjectUser() {
        return subjectUser;
    }

    public Instant getInstant() {
        return instant;
    }

    public void setInstant(Instant instant) {
        this.instant = instant;
    }

    public GridGraphQLContext withInstant(Instant instant) {
        setInstant(instant);
        return this;
    }

    public GdClientInfo getQueriedClient() {
        return queriedClient;
    }

    public void setQueriedClient(GdClientInfo queriedClient) {
        this.queriedClient = queriedClient;
    }

    public GridGraphQLContext withQueriedClient(GdClientInfo queriedClient) {
        setQueriedClient(queriedClient);
        return this;
    }

    public Map<String, GdClientInfo> getClientInfoByLogin() {
        return clientInfoByLogin;
    }

    public FetchedFieldsResolver getFetchedFieldsReslover() {
        return fetchedFieldsReslover;
    }

    public void setFetchedFieldsReslover(FetchedFieldsResolver fetchedFieldsReslover) {
        this.fetchedFieldsReslover = fetchedFieldsReslover;
    }

    public GridGraphQLContext withFetchedFieldsReslover(FetchedFieldsResolver fetchedFieldsReslover) {
        setFetchedFieldsReslover(fetchedFieldsReslover);
        return this;
    }

    /**
     * Вместо этого метода лучше вызывать
     * {@link CampaignInfoService#getAllCampaigns(ClientId)},
     * чтобы не получить NPE.
     * DIRECT-112218
     */
    @Nullable
    public ImmutableList<GdiCampaign> getClientGdiCampaigns() {
        return clientGdiCampaigns;
    }

    public void setClientGdiCampaigns(ImmutableList<GdiCampaign> clientGdiCampaigns) {
        this.clientGdiCampaigns = clientGdiCampaigns;
    }

    public GridGraphQLContext withClientGdiCampaigns(ImmutableList<GdiCampaign> clientGdiCampaigns) {
        setClientGdiCampaigns(clientGdiCampaigns);
        return this;
    }

    @Nullable
    public List<GdiBaseCampaign> getClientGdiBaseCampaigns() {
        return clientGdiBaseCampaigns;
    }

    public void setClientGdiBaseCampaigns(List<GdiBaseCampaign> clientGdiBaseCampaigns) {
        this.clientGdiBaseCampaigns = clientGdiBaseCampaigns;
    }

    public GridGraphQLContext withClientGdiBaseCampaigns(List<GdiBaseCampaign> clientGdiBaseCampaigns) {
        setClientGdiBaseCampaigns(clientGdiBaseCampaigns);
        return this;
    }

    @Nullable
    public Map<Long, GdiCampaign> getGdiCampaignsMap() {
        return gdiCampaignsMap;
    }

    public void setGdiCampaignsMap(@Nullable Map<Long, GdiCampaign> gdiCampaignsMap) {
        this.gdiCampaignsMap = gdiCampaignsMap == null ? null :
                new HashMap<>(gdiCampaignsMap);
    }

    public Map<Long, GdAdGroup> getGdAdGroupsMap() {
        return gdAdGroupsMap;
    }

    public void setGdAdGroups(List<GdAdGroup> gdAdGroups) {
        this.gdAdGroupsMap = gdAdGroups == null ? null :
                unmodifiableMap(listToMap(gdAdGroups, GdAdGroup::getId));
    }

    /**
     * Может быть null, если никто не проставил через set-ер. Такое особенно вероятно, если в ручке используется
     * GridCacheService, т.к. mainAdsByAdGroupId не хранится в кэше.
     */
    @Nullable
    public Map<Long, BannerWithSystemFields> getMainAdsByAdGroupId() {
        return mainAdsByAdGroupId;
    }

    public void setMainAdsByAdGroupId(Map<Long, BannerWithSystemFields> mainAdsByAdGroupId) {
        this.mainAdsByAdGroupId = mainAdsByAdGroupId;
    }

    public Set<String> getResolversToLogValidationResult() {
        return Set.copyOf(resolversToLogValidationResult);
    }

    public void addResolverToLogValidationResult(String resolverName) {
        resolversToLogValidationResult.add(resolverName);
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder("ru.yandex.direct.grid.processing.context.container.GridGraphQLContext{");
        sb.append("instant=").append(instant);
        sb.append(", operator=").append(operator);
        sb.append(", subjectUser=").append(subjectUser);
        sb.append(", queriedClient=").append(queriedClient);
        sb.append(", clientInfoByLogin=").append(clientInfoByLogin);
        sb.append(", clientGdiCampaigns=").append(clientGdiCampaigns);
        sb.append(", gdAdGroupsMap=").append(gdAdGroupsMap);
        sb.append(", mainAdsByAdGroupId=").append(mainAdsByAdGroupId);
        sb.append(", fetchedFieldsReslover=").append(fetchedFieldsReslover);
        sb.append(", isResponseLoggingOnValidationErrorsEnabled=").append(resolversToLogValidationResult);
        sb.append('}');
        return sb.toString();
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }
        GridGraphQLContext that = (GridGraphQLContext) o;
        return Objects.equals(instant, that.instant)
                && Objects.equals(operator, that.operator)
                && Objects.equals(subjectUser, that.subjectUser)
                && Objects.equals(queriedClient, that.queriedClient)
                && Objects.equals(clientInfoByLogin, that.clientInfoByLogin)
                && Objects.equals(clientGdiCampaigns, that.clientGdiCampaigns)
                && Objects.equals(gdAdGroupsMap, that.gdAdGroupsMap)
                && Objects.equals(mainAdsByAdGroupId, that.mainAdsByAdGroupId)
                && Objects.equals(fetchedFieldsReslover, that.fetchedFieldsReslover)
                && Objects.equals(resolversToLogValidationResult,
                that.resolversToLogValidationResult);
    }

    @Override
    public int hashCode() {
        return Objects.hash(
                instant,
                operator,
                subjectUser,
                queriedClient,
                clientInfoByLogin,
                clientGdiCampaigns,
                gdAdGroupsMap,
                mainAdsByAdGroupId,
                fetchedFieldsReslover,
                resolversToLogValidationResult
        );
    }
}
