package ru.yandex.direct.grid.processing.model.promoextension

import io.leangen.graphql.annotations.GraphQLIgnore
import io.leangen.graphql.annotations.GraphQLNonNull
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsPrefix
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsStatusmoderate
import ru.yandex.direct.dbschema.ppc.enums.PromoactionsType
import java.time.LocalDate
import ru.yandex.direct.core.entity.promoextension.model.PromoExtensionUnit
import ru.yandex.direct.dbutil.model.ClientId
import ru.yandex.direct.grid.model.Order
import ru.yandex.direct.grid.processing.model.GdLimitOffset
import ru.yandex.direct.grid.processing.service.cache.CacheFilterData
import ru.yandex.direct.grid.processing.service.cache.CacheRecordInfo
import ru.yandex.direct.grid.processing.service.cache.CachedGridData
import ru.yandex.direct.multitype.entity.LimitOffset

data class GdPromoExtension(
    val id: Long,
    val type: GdPromoExtensionType,
    val amount: Long?,
    val unit: GdPromoExtensionUnit?,
    val prefix: GdPromoExtensionPrefix?,
    val href: String?,
    val description: String,
    val startDate: LocalDate?,
    val finishDate: LocalDate?,
    val name: String, //имя для отображения и сортировки на гриде
    @field:GraphQLIgnore val statusModerate: PromoactionsStatusmoderate,
)

enum class GdPromoExtensionType(val dbType: PromoactionsType) {
    DISCOUNT(PromoactionsType.discount),
    PROFIT(PromoactionsType.profit),
    CASHBACK(PromoactionsType.cashback),
    GIFT(PromoactionsType.gift),
    FREE(PromoactionsType.free),
    INSTALLMENT(PromoactionsType.installment),
    ;

    companion object {
        private val byPromoactionType: Map<PromoactionsType, GdPromoExtensionType> = values().associateBy { it.dbType }

        fun fromPromoactionsType(promoactionsType: PromoactionsType) = byPromoactionType[promoactionsType]
            ?: throw IllegalArgumentException("Invalid promoextension type: $promoactionsType")
    }
}

enum class GdPromoExtensionPrefix(val dbPrefix: PromoactionsPrefix) {
    FROM(PromoactionsPrefix.from),
    TO(PromoactionsPrefix.to),
    ;

    companion object {
        private val byPromoactionPrefix: Map<PromoactionsPrefix, GdPromoExtensionPrefix> =
            values().associateBy { it.dbPrefix }

        fun fromPromoactionsPrefix(promoactionsPrefix: PromoactionsPrefix) = byPromoactionPrefix[promoactionsPrefix]
            ?: throw IllegalArgumentException("Invalid promoextension prefix: $promoactionsPrefix")
    }
}

enum class GdPromoExtensionUnit(val coreValue: PromoExtensionUnit) {
    RUB(PromoExtensionUnit.RUB),
    PCT(PromoExtensionUnit.PCT),
    ;

    companion object {
        private val byCoreValue: Map<PromoExtensionUnit, GdPromoExtensionUnit> = values().associateBy { it.coreValue }

        fun fromCoreValue(promoactionUnit: PromoExtensionUnit) = byCoreValue[promoactionUnit]
            ?: throw IllegalArgumentException("Invalid gd promoextension unit: $promoactionUnit")
    }
}

enum class GdPromoExtensionStatus {
    ON_MODERATION,
    ACCEPTED,
    DECLINED,
    ;
}

enum class GdPromoExtensionPrimaryStatus {
    ON_MODERATION,
    ACTIVE,
    NO_ACTIVE_CAMPAIGNS,
    DECLINED,
    FINISHED,
    ;
}

enum class GdPromoExtensionAction {
    EDIT_PROMO_EXTENSION,
    DELETE_PROMO_EXTENSION,
    ;
}

data class GdPromoExtensionAssociatedCampaign(
    val cid: Long,
    val name: String,
)

data class GdPromoExtensionContext (
    val totalCount: Int,
    val promoExtensionIds: List<@GraphQLNonNull Long>,
    val filter: GdPromoExtensionFilter,
    override var rowset: List<@GraphQLNonNull GdPromoExtension>,
    override var cacheKey: String?,
): CachedGridData<GdPromoExtension>

data class GdPromoExtensionsContainer (
    val filter: GdPromoExtensionFilter?,
    val filterKey: String?,
    val orderBy: List<@GraphQLNonNull GdPromoExtensionOrderBy>?,
    val limitOffset: GdLimitOffset,
    val cacheKey: String?,
)

data class GdPromoExtensionFilter (
    val promoExtensionIdIn: Set<@GraphQLNonNull Long>? = null,
    val nameContains: String? = null,
    val promoExtensionIdContainsAny: Set<@GraphQLNonNull String>? = null,
)

data class GdPromoExtensionOrderBy (
    val field: GdPromoExtensionOrderByField,
    val order: Order,
)

enum class GdPromoExtensionOrderByField {
    ID,
    NAME
}

class PromoExtensionsCacheRecordInfo : CacheRecordInfo<GdPromoExtension, PromoExtensionsCacheFilterData, GdPromoExtensionContext> {
    constructor() : super() {
        // for jackson
    }

    constructor(
        clientId: Long, key: String?,
        filter: PromoExtensionsCacheFilterData) : super(clientId, key, filter) {
    }
}

/**
 *
 * Данные для кэширования промоакций
 */
data class PromoExtensionsCacheFilterData (
    val filter: GdPromoExtensionFilter,
    val orderBy: List<GdPromoExtensionOrderBy>,
): CacheFilterData

/**
 * Подготовленный инпут с фронта
 * Не должен торчать в схеме в качестве элемента
 */
data class PreparedPromoExtensionQueryInput(
    val filter: GdPromoExtensionFilter,
    val orderBy: List<GdPromoExtensionOrderBy>,
    val range: LimitOffset,
) {
    fun toPromoExtensionsCacheRecordInfo(clientId: ClientId, cacheKey: String?) = PromoExtensionsCacheRecordInfo(
        clientId.asLong(),
        cacheKey,
        PromoExtensionsCacheFilterData(
            filter = filter,
            orderBy = orderBy
        ),
    )
}
