package ru.yandex.direct.grid.processing.processor;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import graphql.ExecutionInput;
import graphql.ExecutionResult;
import graphql.GraphQL;
import graphql.execution.ResultPath;
import graphql.execution.instrumentation.Instrumentation;
import graphql.execution.instrumentation.dataloader.DataLoaderDispatcherInstrumentation;
import graphql.execution.instrumentation.dataloader.DataLoaderDispatcherInstrumentationOptions;
import graphql.schema.GraphQLSchema;
import graphql.schema.visibility.NoIntrospectionGraphqlFieldVisibility;
import io.leangen.graphql.GraphQLSchemaGenerator;
import io.leangen.graphql.generator.mapping.common.EnumMapper;
import io.leangen.graphql.generator.mapping.common.ListMapper;
import io.leangen.graphql.generator.mapping.common.NonNullMapper;
import org.dataloader.DataLoaderRegistry;
import org.springframework.aop.support.AopUtils;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.env.EnvironmentType;
import ru.yandex.direct.grid.model.admin.GraphqlSchemaProvider;
import ru.yandex.direct.grid.model.mapper.InputEnumMapper;
import ru.yandex.direct.grid.model.mapper.ListMapperCustom;
import ru.yandex.direct.grid.model.mapper.NonNullMapperCustom;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.processor.interceptor.GridResolverInterceptor;
import ru.yandex.direct.grid.processing.processor.util.GridRequestedFieldsInjector;
import ru.yandex.direct.grid.processing.util.MoneyTypeAdapter;
import ru.yandex.direct.tracing.TraceHelper;

import static com.google.common.base.Preconditions.checkNotNull;
import static graphql.ExecutionInput.newExecutionInput;
import static java.util.Arrays.asList;
import static java.util.Collections.emptyMap;
import static ru.yandex.direct.grid.processing.processor.GridTracingInstrumentationPredicates.fieldPathInSet;
import static ru.yandex.direct.grid.processing.processor.GridTracingInstrumentationPredicates.getPathsToTrace;

/**
 * Обработчик GraphQL-запросов
 */
@ParametersAreNonnullByDefault
public class GridGraphQLProcessor implements GraphqlSchemaProvider {
    private final GraphQL graphQL;
    private final String apiName;
    private final DataLoaderRegistry dataLoaderRegistry;

    /**
     * Основной конструктор, получает список объектов и добавляет все их как синглтоны-обработчики в схему graphql
     */
    public GridGraphQLProcessor(Collection<?> graphQlServices, TraceHelper traceHelper,
                                EnvironmentType environmentType, String tracingServicePrefix,
                                LettuceConnectionProvider lettuce,
                                DataLoaderRegistry dataLoaderRegistry,
                                GridResolverInterceptor resolverInterceptor, String apiName) {
        GraphQLSchema schema = prepareSpqrSchema(graphQlServices, environmentType, resolverInterceptor);
        GridTracingInstrumentation tracingInstrumentation =
                createTracingInstrumentation(graphQlServices, traceHelper, tracingServicePrefix);

       GridRateLimitInstrumentation gridRateLimitInstrumentation =
               createGridRateLimitInstrumentation(graphQlServices, lettuce);

        DataLoaderDispatcherInstrumentationOptions options =
                DataLoaderDispatcherInstrumentationOptions.newOptions();
        DataLoaderDispatcherInstrumentation dispatcherInstrumentation =
                new DataLoaderDispatcherInstrumentation(options);

        List<Instrumentation> instrumentations =
                asList(tracingInstrumentation, dispatcherInstrumentation, gridRateLimitInstrumentation);
        SafeChainedInstrumentation safeChainedInstrumentation = new SafeChainedInstrumentation(instrumentations);
        graphQL = GraphQL.newGraphQL(schema)
                .instrumentation(safeChainedInstrumentation)
                .build();
        this.apiName = apiName;
        this.dataLoaderRegistry = dataLoaderRegistry;
    }

    /**
     * Создаём инструментацию {@link GridTracingInstrumentation} для трасировки запросов к отдельным полям.
     * Она одна на приложение.
     */
    private static GridTracingInstrumentation createTracingInstrumentation(Collection<?> graphQlServices,
                                                                           TraceHelper traceHelper,
                                                                           String tracingServicePrefix) {
        Set<ResultPath> pathsToTrace = getPathsToTrace(graphQlServices);
        return new GridTracingInstrumentation(traceHelper, fieldPathInSet(pathsToTrace), tracingServicePrefix);
    }

    /**
     * Создаём инструментацию {@link GridRateLimitInstrumentation} для ограничение количества запросов к резолверам.
     * Она одна на приложение.
     */
    private static GridRateLimitInstrumentation createGridRateLimitInstrumentation(Collection<?> graphQlServices,
                                                                                   LettuceConnectionProvider lettuce) {
        Map<ResultPath, GraphQLRateLimit> methods = getRateLimitMethodsWithAnnotation(graphQlServices);
        return new GridRateLimitInstrumentation(lettuce, methods);
    }

    private static Map<ResultPath, GraphQLRateLimit> getRateLimitMethodsWithAnnotation(Collection<?> graphQlServices) {
        return graphQlServices.stream()
                .map(AopUtils::getTargetClass)
                .flatMap(clazz -> Arrays.stream(clazz.getDeclaredMethods())
                        .filter(m -> m.isAnnotationPresent(GraphQLRateLimit.class)))
                .collect(Collectors.toMap(m -> ResultPath.fromList(List.of(m.getName())),
                        m -> m.getAnnotation(GraphQLRateLimit.class)));
    }

    /**
     * Создает GraphQL схему. На проде и на ТС не позволяем получать параметры схемы
     */
    static GraphQLSchema prepareSpqrSchema(Collection<?> graphQlServices, EnvironmentType environmentType,
                                           GridResolverInterceptor resolverInterceptor) {
        /*
          Список пакетов состоит из моделей, в которых присутствуют аннотации graphql
          и моделей использующихся как GraphQLArgument
         */
        GraphQLSchemaGenerator graphQLSchemaGenerator = new GraphQLSchemaGenerator()
                .withTypeMappers((conf, current) ->
                        current
                                .replace(EnumMapper.class, m -> new InputEnumMapper())
                                .replace(NonNullMapper.class, m -> new NonNullMapperCustom())
                                .replace(ListMapper.class, m -> new ListMapperCustom())
                )
                .withBasePackages("ru.yandex.direct.grid.processing.model",
                        "ru.yandex.direct.grid.core.entity",
                        "ru.yandex.direct.grid.model")
                .withTypeAdapters(new MoneyTypeAdapter())
                .withArgumentInjectors(new GridRequestedFieldsInjector())
                .withTypeInfoGenerator(new NamePatchingTypeInfoGenerator())
                .withResolverInterceptors(resolverInterceptor);

        graphQlServices.forEach(s -> graphQLSchemaGenerator.withOperationsFromSingleton(s, AopUtils.getTargetClass(s)));

        GraphQLSchema schema = graphQLSchemaGenerator.generate();
        if (environmentType.isProductionOrPrestable() || EnvironmentType.TESTING.equals(environmentType)) {
            schema = GraphQLSchema.newSchema(schema)
                    .fieldVisibility(NoIntrospectionGraphqlFieldVisibility.NO_INTROSPECTION_FIELD_VISIBILITY)
                    .build();
        }

        return schema;
    }

    /**
     * Обработать запрос к graphql и вернуть результат
     *
     * @param operationName название операции
     * @param query         строка с graphql-запросом
     * @param arguments     аргументы
     * @param context       контекст исполнения запроса, должен иметь заполненное поле с описание оператора запроса
     */
    public ExecutionResult processQuery(@Nullable String operationName, String query,
                                        @Nullable Map<String, Object> arguments, GridGraphQLContext context) {
        checkNotNull(context, "context cannot be null");
        checkNotNull(context.getOperator(), "operator cannot be null");
        context.setInstant(Instant.now());

        GraphQL graphQL = getGraphql();

        ExecutionInput executionInput = newExecutionInput()
                .operationName(operationName)
                .query(query)
                .dataLoaderRegistry(dataLoaderRegistry)
                .variables(arguments != null ? arguments : emptyMap())
                .context(context)
                .build();
        return graphQL.execute(executionInput);
    }

    @Override
    public GraphQL getGraphql() {
        return graphQL;
    }

    @Override
    public String getGraphqlApiName() {
        return apiName;
    }
}
