package ru.yandex.direct.grid.processing.processor;

import java.util.List;
import java.util.function.Predicate;

import javax.annotation.ParametersAreNonnullByDefault;

import graphql.execution.instrumentation.InstrumentationContext;
import graphql.execution.instrumentation.SimpleInstrumentation;
import graphql.execution.instrumentation.SimpleInstrumentationContext;
import graphql.execution.instrumentation.parameters.InstrumentationExecutionParameters;
import graphql.execution.instrumentation.parameters.InstrumentationFieldFetchParameters;
import graphql.execution.instrumentation.parameters.InstrumentationValidationParameters;
import graphql.language.Document;
import graphql.validation.ValidationError;

import ru.yandex.direct.tracing.Trace;
import ru.yandex.direct.tracing.TraceGuard;
import ru.yandex.direct.tracing.TraceHelper;
import ru.yandex.direct.tracing.TraceProfile;
import ru.yandex.direct.tracing.real.RealTrace;
import ru.yandex.direct.tracing.util.TraceUtil;

import static graphql.execution.instrumentation.SimpleInstrumentationContext.whenCompleted;
import static ru.yandex.direct.grid.processing.processor.util.InstrumentationUtil.getMethodName;


/**
 * Инструментарий, профилирующий все запросы к полям, которые подходят по переданному предикату
 */
@ParametersAreNonnullByDefault
public class GridTracingInstrumentation extends SimpleInstrumentation {
    private final TraceHelper traceHelper;
    private final String tracingServicePrefix;
    private final Predicate<InstrumentationFieldFetchParameters> fieldTracingPredicate;
    private final boolean traceFieldWithGuard;

    GridTracingInstrumentation(TraceHelper traceHelper,
                               Predicate<InstrumentationFieldFetchParameters> fieldTracingPredicate, String tracingServicePrefix) {
        this(traceHelper, fieldTracingPredicate, tracingServicePrefix, true);
    }

    GridTracingInstrumentation(TraceHelper traceHelper,
                               Predicate<InstrumentationFieldFetchParameters> fieldTracingPredicate, String tracingServicePrefix,
                               boolean traceFieldWithGuard) {
        this.traceHelper = traceHelper;
        this.tracingServicePrefix = tracingServicePrefix;
        this.fieldTracingPredicate = fieldTracingPredicate;
        this.traceFieldWithGuard = traceFieldWithGuard;
    }

    @Override
    public InstrumentationContext<Object> beginFieldFetch(InstrumentationFieldFetchParameters parameters) {
        if (!fieldTracingPredicate.test(parameters)) {
            return new SimpleInstrumentationContext<>();
        }

        String method = tracingServicePrefix + "." + getMethodName(parameters);
        if (traceFieldWithGuard) {
            TraceProfile topProfile = Trace.current().profile("graphql." + method);
            Trace trace = RealTrace.builder()
                    .withIds(Trace.current().getTraceId(), Trace.current().getSpanId(), TraceUtil.randomId())
                    .withService(traceHelper.getService())
                    .withMethod(method)
                    .build();
            TraceGuard traceGuard = traceHelper.guard(trace);
            return whenCompleted((result, t) -> {
                topProfile.close();
                traceGuard.close();
            });

        } else {
            TraceProfile profile = Trace.current().profile(method);
            return whenCompleted((result, t) -> profile.close());
        }
    }

    @Override
    public InstrumentationContext<Document> beginParse(InstrumentationExecutionParameters parameters) {
        TraceProfile profile = Trace.current().profile("graphql:parse");
        return whenCompleted((result, t) -> profile.close());
    }

    @Override
    public InstrumentationContext<List<ValidationError>> beginValidation(
            InstrumentationValidationParameters parameters) {
        TraceProfile profile = Trace.current().profile("graphql:validation");
        return whenCompleted((result, t) -> profile.close());
    }

}
