package ru.yandex.direct.grid.processing.processor;

import java.lang.reflect.Method;
import java.lang.reflect.Parameter;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.function.Predicate;
import java.util.stream.Collectors;

import javax.annotation.Nullable;

import com.google.common.collect.Multimap;
import com.google.common.collect.Multimaps;
import graphql.execution.ResultPath;
import graphql.execution.instrumentation.parameters.InstrumentationFieldFetchParameters;
import io.leangen.graphql.annotations.GraphQLContext;
import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLQuery;
import one.util.streamex.StreamEx;
import org.springframework.aop.support.AopUtils;

import static java.lang.String.format;
import static java.util.Collections.singletonList;

class GridTracingInstrumentationPredicates {
    private static final String NO_PARENT = "";

    private GridTracingInstrumentationPredicates() {
    }

    /**
     * Вернуть предикат, который возвращает истину при проверке запроса к полю с полным путем, содержащимся в заданном
     * наборе
     *
     * @param paths набор корректных полных путей к graphQl-полям
     */
    static Predicate<InstrumentationFieldFetchParameters> fieldPathInSet(Set<ResultPath> paths) {
        return p -> paths.contains(p.getExecutionStepInfo().getPath());
    }

    /**
     * Просмотреть все сервисы, обрабатывающие GraphQl-запросы, и вернуть набор полных путей полей, помеченных в этих
     * сервисах как {@link GraphQLQuery} или {@link GraphQLMutation}. Такие поля нам нужны так как они в большинстве
     * случаев тождественны контроллерам и должны профилироваться каждый по отдельности.
     */
    static Set<ResultPath> getPathsToTrace(Collection<?> graphQlServices) {
        List<QueryData> queryData = graphQlServices.stream()
                .map(AopUtils::getTargetClass)
                .flatMap(clazz -> Arrays.stream(clazz.getDeclaredMethods())
                        .filter(m -> m.isAnnotationPresent(GraphQLQuery.class)
                                || m.isAnnotationPresent(GraphQLMutation.class))
                        .map(GridTracingInstrumentationPredicates::queryDataFromMethod))
                .collect(Collectors.toList());

        return getPathSet(queryData);
    }

    /**
     * Пробегаемся по дереву зависимостей между получателями данных {@code queryData} и составляем перечень путей,
     * которые хотим инструментировать.
     */
    static Set<ResultPath> getPathSet(List<QueryData> queryData) {
        Multimap<String, QueryData> dataByParent = Multimaps.index(queryData,
                data -> data.parentType != null ? data.parentType.getTypeName() : NO_PARENT);
        Set<ResultPath> result = new HashSet<>();

        for (QueryData root : dataByParent.get(NO_PARENT)) {
            List<List<String>> res = digDown(root, dataByParent);
            StreamEx.of(res)
                    .map(ResultPath::fromList)
                    .forEach(result::add);
        }
        return result;
    }

    /**
     * От указанного {@code root} строит иерархию зависимостей на основе {@code queryDataByParent}.
     * Для каждого узла (листового и промежуточного) возвращается список
     * {@code List<String>}, содержащий путь от корня до этого узла.
     */
    private static List<List<String>> digDown(QueryData root, Multimap<String, QueryData> queryDataByParent) {
        String lowLevelParentType = root.returnType.getTypeName();
        Collection<QueryData> children = queryDataByParent.get(lowLevelParentType);
        List<List<String>> result = new ArrayList<>();
        result.add(singletonList(root.name));
        for (QueryData child : children) {
            List<List<String>> childPaths = digDown(child, queryDataByParent);
            for (List<String> childPath : childPaths) {
                result.add(StreamEx.of(root.name).append(childPath).toList());
            }
        }
        return result;
    }

    private static QueryData queryDataFromMethod(Method method) {
        String name;
        if (method.isAnnotationPresent(GraphQLQuery.class)) {
            GraphQLQuery query = method.getAnnotation(GraphQLQuery.class);
            name = query.name();
        } else if (method.isAnnotationPresent(GraphQLMutation.class)) {
            GraphQLMutation mutation = method.getAnnotation(GraphQLMutation.class);
            name = mutation.name();
        } else {
            throw new IllegalArgumentException(format("Unknown method %s for get query data", method.getName()));
        }

        Type returnType = method.getGenericReturnType();
        Type context = Arrays.stream(method.getParameters())
                .filter(p -> p.isAnnotationPresent(GraphQLContext.class))
                .map(Parameter::getParameterizedType)
                .findFirst()
                .orElse(null);

        return new QueryData(name, returnType, context);
    }

    static class QueryData {
        private final String name;
        private final Type returnType;
        private final Type parentType;

        QueryData(String name, Type returnType, @Nullable Type parentType) {
            this.name = name;
            this.returnType = returnType;
            this.parentType = parentType;
        }
    }
}
