package ru.yandex.direct.grid.processing.processor;

import java.time.Clock;

import javax.annotation.ParametersAreNonnullByDefault;

import graphql.execution.instrumentation.parameters.InstrumentationFieldFetchParameters;
import io.leangen.graphql.annotations.GraphQLQuery;
import io.lettuce.core.SetArgs;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;

import ru.yandex.direct.common.lettuce.LettuceConnectionProvider;
import ru.yandex.direct.common.lettuce.LettuceExecuteException;
import ru.yandex.direct.web.core.exception.RateLimitExceededException;

import static ru.yandex.direct.common.configuration.RedisConfiguration.LETTUCE;


/**
 * {@link RateLimitHandler} ограничивает частоту запросов к резолверам в GraphGL:
 * можно делать не больше <i>maxRequests</i> запросов за <i>periodInSeconds</i> секунд.
 * <p>
 * Чтобы этот компонент заработал для метода, его надо пометить аннотацией, например:
 * <code>
 * {@literal @}GraphQLRateLimit(periodInSeconds = 60, maxRequests = 20) {
 * ...
 * }
 * </code>
 * <p>
 * Аннотацию можно указывать только один раз на каждый резолвер.
 * <p>
 * Важно, что имя аннотируемого метода должно совпадать с именем GraphQL ручки
 * (см. параметр {@link GraphQLQuery#name()} и реализацию
 * {@link GridRateLimitInstrumentation#beginFieldFetch(InstrumentationFieldFetchParameters)}).
 */
@ParametersAreNonnullByDefault
public class RateLimitHandler {
    private static final Logger logger = LoggerFactory.getLogger(RateLimitHandler.class);

    private final Clock clock;
    private LettuceConnectionProvider lettuce;

    private RateLimitHandler(@Qualifier(LETTUCE) LettuceConnectionProvider lettuce, Clock clock) {
        this.lettuce = lettuce;
        this.clock = clock;
    }

    RateLimitHandler(@Qualifier(LETTUCE) LettuceConnectionProvider connectionProvider) {
        this(connectionProvider, Clock.systemDefaultZone());
    }

    void handleRequest(Long clientId, String methodName, GraphQLRateLimit graphQLRateLimit)
            throws RateLimitExceededException {

        checkRateLimits(graphQLRateLimit, methodName, clientId);

    }

    // Метод скопирован из RateLimitInterceptor
    private long incrAndGetRedisKey(String redisKey, long expireAfterSecs) {
        // Устанавливаем начальное кол-во запросов, если еще не устанавливали ранее (NX),
        // с expiration-ом (EX) через expireAfterSecs секунд

        lettuce.call("redis:set",
                com -> com.set(redisKey, "0", SetArgs.Builder.nx().ex(expireAfterSecs))
        );
        return lettuce.call("redis:incr", com -> com.incr(redisKey));
    }

    /**
     * Увеличить кол-во сделанных запросов и вернуть текущее значение, начиная с момента startOfPeriodInSec
     *
     * Метод скопирован из RateLimitInterceptor
     */
    private long incrAndGetRequestMade(String methodName, long clientId, GraphQLRateLimit rateLimit,
                                       long startOfPeriodInSec) {
        String redisKey = String.format("web-ratelimit-%s-%s-%s-%s-%s",
                "grid",
                methodName,
                rateLimit.periodInSeconds(),
                clientId,
                startOfPeriodInSec);

        try {
            return incrAndGetRedisKey(redisKey, rateLimit.periodInSeconds());
        } catch (LettuceExecuteException e) {
            logger.error("Can't increment and get redisKey: {}", redisKey, e);
            // Считаем что отказ Redis-а должен быть относительно нечастым событием, поэтому выбираем
            // счастье пользователей, разрешая им немного превысить ограничение по кол-ву запросов
            return 0;
        }
    }

    // Метод скопирован из RateLimitInterceptor
    private void checkRateLimits(GraphQLRateLimit rateLimit, String methodName, long clientId) {
        long currentTime = clock.instant().getEpochSecond();

        /*
         * Эта реализация проверяет, что клиент делает не больше rateLimit.maxRequests()
         * в окне времени от startOfPeriod до startOfPeriod + rateLimit.periodInSeconds().
         * Это окно времени меняется каждые rateLimit.periodInSeconds() секунд.
         */
        long startOfPeriod = currentTime - currentTime % rateLimit.periodInSeconds();

        long requestsMade = incrAndGetRequestMade(methodName, clientId, rateLimit, startOfPeriod);
        if (requestsMade > rateLimit.maxRequests()) {
            throw new RateLimitExceededException("Rate limit exceeded for client " + clientId);
        }
    }
}
