package ru.yandex.direct.grid.processing.processor.interceptor;

import java.lang.annotation.Annotation;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.annotations.GraphQLNonNull;
import io.leangen.graphql.execution.InvocationContext;
import io.leangen.graphql.execution.ResolverInterceptor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.grid.processing.annotations.EnableLoggingOnValidationIssues;
import ru.yandex.direct.grid.processing.context.container.GridGraphQLContext;
import ru.yandex.direct.grid.processing.exception.GridSkipInErrorsException;
import ru.yandex.direct.grid.processing.service.attributes.AttributeResolverService;
import ru.yandex.direct.web.annotations.RequiredAttributes;

@Component
@ParametersAreNonnullByDefault
public class GridResolverInterceptor implements ResolverInterceptor {

    private final AttributeResolverService attributeResolverService;
    private final OperatorOrUserIsBlockedChecker operatorOrUserIsBlockedChecker;

    @Autowired
    public GridResolverInterceptor(AttributeResolverService attributeResolverService,
                                   OperatorOrUserIsBlockedChecker operatorOrUserIsBlockedChecker) {
        this.attributeResolverService = attributeResolverService;
        this.operatorOrUserIsBlockedChecker = operatorOrUserIsBlockedChecker;
    }

    @Override
    public Object aroundInvoke(InvocationContext context, Continuation continuation) throws Exception {
        operatorOrUserIsBlockedChecker.checkResolver(context);
        checkRequiredAttributes(context);
        checkNeedLoggingOnValidationIssues(context);
        return continuation.proceed(context);
    }

    private void checkRequiredAttributes(InvocationContext context) {
        RequiredAttributes requiredAttributes = getAnnotation(context, RequiredAttributes.class);

        if (requiredAttributes != null && requiredAttributes.value().length > 0) {
            if (!attributeResolverService.resolve(requiredAttributes.value())) {
                GraphQLNonNull requiredField = getAnnotation(context, GraphQLNonNull.class);
                if (requiredField != null) {
                    // поле должно быть nullable, если у нее есть RequiredAttributes
                    throw new IllegalStateException(String.format("field %s with @RequiredAttributes must be nullable",
                            context.getResolver().getOperationName()));
                }

                // ошибку фронту не покажем, но в поле значение не вернем - будет null
                throw new GridSkipInErrorsException("No rights for field " + context.getResolver().getOperationName());
            }
        }
    }

    private void checkNeedLoggingOnValidationIssues(InvocationContext context) {
        EnableLoggingOnValidationIssues enableLoggingOnValidationIssuesAnnotation =
                getAnnotation(context, EnableLoggingOnValidationIssues.class);

        if (null != enableLoggingOnValidationIssuesAnnotation) {
            GraphQLMutation mutationAnnotation = getAnnotation(context, GraphQLMutation.class);
            if (null != mutationAnnotation) {
                ((GridGraphQLContext) context.getResolutionEnvironment().dataFetchingEnvironment.getContext())
                        .addResolverToLogValidationResult(mutationAnnotation.name());
            }
        }
    }

    private static <T extends Annotation> T getAnnotation(InvocationContext context, Class<T> clazz) {
        return context.getResolver().getExecutable().getDelegate().getAnnotation(clazz);
    }
}
