package ru.yandex.direct.grid.processing.processor.interceptor;

import java.lang.annotation.Annotation;

import javax.annotation.ParametersAreNonnullByDefault;

import io.leangen.graphql.annotations.GraphQLMutation;
import io.leangen.graphql.execution.InvocationContext;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.stereotype.Service;

import ru.yandex.direct.core.entity.user.model.User;
import ru.yandex.direct.core.security.AccessDeniedException;
import ru.yandex.direct.core.security.SecurityTranslations;
import ru.yandex.direct.web.annotations.AllowedBlockedOperatorOrUser;
import ru.yandex.direct.web.core.security.DirectWebAuthenticationSource;

import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.checkOperatorIsBlocked;
import static ru.yandex.direct.core.entity.user.utils.BlockedUserUtil.checkSubjectUserIsBlocked;

@Service
@ParametersAreNonnullByDefault
class OperatorOrUserIsBlockedChecker {

    private final DirectWebAuthenticationSource directWebAuthenticationSource;

    @Autowired
    public OperatorOrUserIsBlockedChecker(DirectWebAuthenticationSource directWebAuthenticationSource) {
        this.directWebAuthenticationSource = directWebAuthenticationSource;
    }

    /**
     * Проверяет мутации на доступность заблокированному оператору или клиенту
     * Запросы(GraphQLQuery) доступны заблокированному оператору или клиенту
     * Для public-api ничего не проверяет, т.к. нет информации по оператору и клиенту в Директе
     */
    void checkResolver(InvocationContext context) {
        if (!resolverHasAnnotation(context, GraphQLMutation.class)
                || SecurityContextHolder.getContext().getAuthentication() == null) {
            return;
        }

        User operator = directWebAuthenticationSource.getAuthentication().getOperator();
        User subjectUser = directWebAuthenticationSource.getAuthentication().getSubjectUser();
        if (operator == null || subjectUser == null) {
            return;
        }

        boolean resolverAllowedForBlockedOperatorOrUser =
                resolverHasAnnotation(context, AllowedBlockedOperatorOrUser.class);

        if (checkOperatorIsBlocked(operator, resolverAllowedForBlockedOperatorOrUser)) {
            throw new AccessDeniedException(
                    "Operator is blocked for resolver " + context.getResolver().getOperationName(),
                    SecurityTranslations.INSTANCE.loginIsDisabled());
        }

        if (checkSubjectUserIsBlocked(subjectUser, resolverAllowedForBlockedOperatorOrUser, operator)) {
            throw new AccessDeniedException(
                    "Client is blocked for resolver " + context.getResolver().getOperationName(),
                    SecurityTranslations.INSTANCE.loginIsDisabled());
        }
    }

    private static boolean resolverHasAnnotation(InvocationContext context, Class<? extends Annotation> clazz) {
        return context.getResolver().getExecutable().getDelegate().getAnnotation(clazz) != null;
    }

}
