package ru.yandex.direct.grid.processing.processor.util;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import graphql.language.Field;
import graphql.language.FragmentDefinition;
import graphql.language.FragmentSpread;
import graphql.language.InlineFragment;
import graphql.language.Selection;
import graphql.language.SelectionSet;
import graphql.language.SelectionSetContainer;

@ParametersAreNonnullByDefault
public class FetchedFieldsResolverUtil {

    @Nullable
    static Field getChildField(List<String> path, @Nullable Field rootField,
                               Map<String, FragmentDefinition> fragmentsByName) {
        Field tmpField = rootField;
        for (String s : path) {
            if (tmpField == null) {
                return null;
            }
            tmpField = getChildField(s, tmpField, fragmentsByName);
        }
        return tmpField;
    }

    @Nullable
    public static Field getChildField(String childName,
                                      @Nullable SelectionSetContainer selectionSetContainer,
                                      Map<String, FragmentDefinition> fragmentsByName) {
        if (selectionSetContainer != null) {
            List<Field> fields = getFields(selectionSetContainer.getSelectionSet(), fragmentsByName);

            return fields.stream()
                    .filter(f -> f.getName().equals(childName))
                    .findFirst()
                    .orElse(null);
        }
        return null;
    }

    /**
     * Возвращает список всех полей с учетом фрагментов
     */
    static List<Field> getFields(SelectionSet selectionSet,
                                 Map<String, FragmentDefinition> fragmentsByName) {
        List<Field> fields = new ArrayList<>(selectionSet.getSelections().size());

        for (Selection selection : selectionSet.getSelections()) {
            if (selection instanceof Field) {
                fields.add((Field) selection);
            } else if (selection instanceof InlineFragment) {
                InlineFragment inlineFragment = (InlineFragment) selection;
                //могут быть вложенные фрагменты, поэтому ищем рекурсивно
                List<Field> inlineFragmentSelections =
                        getFields(inlineFragment.getSelectionSet(), fragmentsByName);

                fields.addAll(inlineFragmentSelections);
            } else if (selection instanceof FragmentSpread) {
                String fragmentName = ((FragmentSpread) selection).getName();
                FragmentDefinition fragmentDefinition = fragmentsByName.get(fragmentName);
                //могут быть вложенные фрагменты, поэтому ищем рекурсивно
                List<Field> fragmentSelections =
                        getFields(fragmentDefinition.getSelectionSet(), fragmentsByName);

                fields.addAll(fragmentSelections);
            } else {
                throw new IllegalStateException("Unsupported selection type: " + selection.getClass().getName());
            }
        }

        return fields;
    }

    public static boolean hasChildField(String name, @Nullable Field field,
                                        Map<String, FragmentDefinition> fragmentsByName) {
        return getChildField(name, field, fragmentsByName) != null;
    }

    public static Set<String> presentChildFields(@Nullable Field field,
                                                 Map<String, FragmentDefinition> fragmentsByName) {
        return presentChildFields(null, field, fragmentsByName);
    }

    public static Set<String> presentChildFields(@Nullable Set<String> names, @Nullable Field field,
                                                 Map<String, FragmentDefinition> fragmentsByName) {

        if (field != null) {
            List<Field> fields = getFields(field.getSelectionSet(), fragmentsByName);

            var result = fields.stream()
                    .map(f -> f.getName())
                    .filter(Objects::nonNull);
            if (names != null) {
                result = result.filter(f -> names.contains(f));
            }
            return result.collect(Collectors.toSet());
        }
        return null;
    }
}
