package ru.yandex.direct.grid.processing.service.adgeneration;

import java.util.List;

import javax.annotation.ParametersAreNonnullByDefault;

import one.util.streamex.EntryStream;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.direct.core.entity.adgroup.container.ComplexTextAdGroup;
import ru.yandex.direct.core.entity.adgroup.service.AdGroupService;
import ru.yandex.direct.core.entity.banner.model.BannerWithAdGroupId;
import ru.yandex.direct.core.entity.banner.model.TextBanner;
import ru.yandex.direct.core.entity.feature.service.FeatureService;
import ru.yandex.direct.core.entity.keyword.model.Keyword;
import ru.yandex.direct.dbutil.model.ClientId;
import ru.yandex.direct.feature.FeatureName;
import ru.yandex.direct.result.MassResult;
import ru.yandex.direct.result.Result;

import static ru.yandex.direct.utils.FunctionalUtils.listToSet;
import static ru.yandex.direct.utils.FunctionalUtils.mapList;

@Component
@ParametersAreNonnullByDefault
public class AdGenerationLoggingService {

    private static final Logger logger = LoggerFactory.getLogger(AdGenerationLoggingService.class);

    private final FeatureService featureService;
    private final AdGroupService adGroupService;

    @Autowired
    public AdGenerationLoggingService(FeatureService featureService,
                                      AdGroupService adGroupService) {
        this.featureService = featureService;
        this.adGroupService = adGroupService;
    }

    public void logGroupKeywords(Long operatorUid, ClientId clientId, List<ComplexTextAdGroup> adGroups,
                                 MassResult<Long> result) {
        if (!result.isSuccessful()) {
            return;
        }

        if (featureService.isEnabled(operatorUid, FeatureName.SUGGEST_GENERATED_GROUP_PHRASES_FOR_OPERATOR) ||
                featureService.isEnabled(operatorUid, FeatureName.SUGGEST_GENERATED_PHRASES_BY_SNIPPET_FOR_OPERATOR) ||
                featureService.isEnabledForClientId(clientId, FeatureName.UC_SUGGEST_AUDIENCE_STEP_PHRASES)) {
            var updatedAdGroups = EntryStream
                    .zip(adGroups, result.getResult())
                    .filterValues(Result::isSuccessful)
                    .mapValues(Result::getResult)
                    .keys()
                    .toList();

            var adGroupIds = mapList(updatedAdGroups, adGroup -> adGroup.getAdGroup().getId());

            var campaignIds = adGroupService.getCampaignIdsByAdgroupIds(clientId, adGroupIds);

            updatedAdGroups.forEach(adGroup -> {
                if (!adGroup.getKeywords().isEmpty()) {
                    var keywords = mapList(adGroup.getKeywords(), Keyword::getPhrase);
                    logger.info("Saved keywords for cid={}: {}",
                            campaignIds.get(adGroup.getAdGroup().getId()),
                            String.join(", ", keywords));
                }
            });
        }
    }

    public <T extends BannerWithAdGroupId> void logBannerTitleAndSnippet(Long operatorUid, ClientId clientId,
                                                                         List<T> banners,
                                                                         MassResult<Long> result) {
        if (!result.isSuccessful()) {
            return;
        }

        if (featureService.isEnabled(operatorUid, FeatureName.SUGGEST_GENERATED_TITLE_AND_SNIPPET_FOR_OPERATOR) ||
                featureService.isEnabledForClientId(clientId, FeatureName.UC_SUGGEST_ADS_STEP_SNIPPET)) {
            var updatedBanners = EntryStream
                    .zip(result.toResultList(), banners)
                    .filterKeys(Result::isSuccessful)
                    .mapKeys(Result::getResult)
                    .selectValues(TextBanner.class)
                    .toMap();

            var adGroupIds = listToSet(updatedBanners.values(), BannerWithAdGroupId::getAdGroupId);

            var campaignIds = adGroupService.getCampaignIdsByAdgroupIds(clientId, adGroupIds);

            updatedBanners.forEach((id, banner) -> {
                Long cid = campaignIds.getOrDefault(banner.getAdGroupId(), 0L);
                logger.info("Saved banner cid={}, pid={}, bid={} with title='{}', snippet='{}'",
                        cid, banner.getAdGroupId(), banner.getId(), banner.getTitle(), banner.getBody());
            });
        }
    }
}
