package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AdGroupStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.adgroup.AggregatedStatusAdGroupData;
import ru.yandex.direct.grid.core.entity.group.model.GdiGroupPrimaryStatus;

import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.DRAFT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_CRIT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_BL_PROCESSING;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_BL_PROCESSING_WITH_OLD_VERSION_SHOWN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_HAS_ADS_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_ON_MODERATION;

@ParametersAreNonnullByDefault
public class AdGroupPrimaryStatusCalculator {
    private static final Logger logger =
            LoggerFactory.getLogger(AdGroupPrimaryStatusCalculator.class);

    private static final Set<GdSelfStatusEnum> MODERATION_ADS_SET = Set.of(DRAFT, STOP_CRIT);
    private static final Set<GdSelfStatusEnum> MODERATION_CONDITIONS_SET = Set.of(STOP_WARN, DRAFT, STOP_CRIT);

    private AdGroupPrimaryStatusCalculator() {
    }

    public static GdiGroupPrimaryStatus convertToPrimaryStatus(@Nullable AggregatedStatusAdGroupData statusData) {
        if (statusData == null || statusData.getStatus().isEmpty()) {
            return GdiGroupPrimaryStatus.DRAFT;
        }
        GdSelfStatusEnum status = statusData.getStatus().get();
        List<GdSelfStatusReason> reasons = ListUtils.emptyIfNull(statusData.getReasons());
        Collection<AdGroupStatesEnum> states = CollectionUtils.emptyIfNull(statusData.getStates());

        // Архивные
        if (isArchivedPrimaryStatus(status, reasons, states)) {
            return GdiGroupPrimaryStatus.ARCHIVED;
        }
        // Обрабатывается: Статуса нет в списке фильтров, отдаем null чтобы не фильтровать по нему
        // При данном кейсе группа будет отображаться в "Все кампании" и "Все кроме архивных"
        //
        // DIRECT-149475: теперь обрабатывающиеся группы будут попадать в "Запущенные"
        if (isProcessingPrimaryStatus(status, reasons, states)) {
            return null;
        }
        // На модерации
        else if (isModerationPrimaryStatus(status, reasons, states)) {
            return GdiGroupPrimaryStatus.MODERATION;
        }
        // Отклоненные
        else if (isRejectedPrimaryStatus(status, reasons, states)) {
            return GdiGroupPrimaryStatus.REJECTED;
        }
        // Остановленные
        else if (isStoppedPrimaryStatus(status, reasons, states)) {
            return GdiGroupPrimaryStatus.STOPPED;
        }
        // Черновики
        else if (isDraftPrimaryStatus(status, reasons, states)) {
            return GdiGroupPrimaryStatus.DRAFT;
        }
        // Запущенные
        else if (isActivePrimaryStatus(status, reasons, states)) {
            return GdiGroupPrimaryStatus.ACTIVE;
        }

        logger.warn("Can't find primary status for {}", statusData);
        return GdiGroupPrimaryStatus.ACTIVE;
    }

    // Архивные
    private static boolean isArchivedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                   Collection<AdGroupStatesEnum> states) {
        return status.equals(GdSelfStatusEnum.ARCHIVED);
    }

    // Обрабатывается
    private static boolean isProcessingPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                     Collection<AdGroupStatesEnum> states) {
        return (status.equals(GdSelfStatusEnum.RUN_WARN) && reasons.contains(ADGROUP_BL_PROCESSING_WITH_OLD_VERSION_SHOWN))
                || (Set.of(DRAFT, STOP_CRIT).contains(status) && reasons.contains(ADGROUP_BL_PROCESSING));
    }

    // На модерации
    private static boolean isModerationPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                     Collection<AdGroupStatesEnum> states) {
        return GdSelfStatusEnum.ON_MODERATION.equals(status)
                || (MODERATION_ADS_SET.contains(status) && reasons.contains(ADGROUP_HAS_ADS_ON_MODERATION))
                || (
                MODERATION_CONDITIONS_SET.contains(status) && reasons.contains(ADGROUP_SHOW_CONDITIONS_ON_MODERATION));
    }

    // Отклоненные
    private static boolean isRejectedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                   Collection<AdGroupStatesEnum> states) {
        return Set.of(DRAFT, STOP_CRIT).contains(status) && reasons.contains(ADGROUP_REJECTED_ON_MODERATION);
    }

    // Остановленные
    private static boolean isStoppedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                  Collection<AdGroupStatesEnum> states) {
        return GdSelfStatusEnum.allStop().contains(status) && status != GdSelfStatusEnum.STOP_PROCESSING;
    }

    // Черновики
    private static boolean isDraftPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                Collection<AdGroupStatesEnum> states) {
        return Objects.equals(status, DRAFT);
    }

    // Запущенные
    private static boolean isActivePrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                 Collection<AdGroupStatesEnum> states) {
        return GdSelfStatusEnum.allRun().contains(status) || status == GdSelfStatusEnum.STOP_PROCESSING;
    }
}
