package ru.yandex.direct.grid.processing.service.aggregatedstatuses;

import java.util.Collection;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import javax.annotation.Nullable;
import javax.annotation.ParametersAreNonnullByDefault;

import org.apache.commons.collections4.ListUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AdStatesEnum;
import ru.yandex.direct.core.entity.aggregatedstatuses.ad.AggregatedStatusAdData;
import ru.yandex.direct.grid.core.entity.banner.model.GdiBannerPrimaryStatus;

import static org.apache.commons.collections.CollectionUtils.containsAny;
import static org.apache.commons.collections4.CollectionUtils.emptyIfNull;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.DRAFT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.RUN_WARN;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.STOP_CRIT;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusEnum.allStop;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.ADGROUP_SHOW_CONDITIONS_SUSPENDED_BY_USER;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_DISPLAY_HREF_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_IMAGE_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_SUSPENDED_BY_USER;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_TURBOLANDING_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_VCARD_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.AD_VIDEO_ADDITION_REJECTED_ON_MODERATION;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.CAMPAIGN_SUSPENDED_BY_USER;
import static ru.yandex.direct.core.entity.aggregatedstatuses.GdSelfStatusReason.REJECTED_ON_MODERATION;

@ParametersAreNonnullByDefault
public class AdPrimaryStatusCalculator {

    private static final Logger logger = LoggerFactory.getLogger(AdPrimaryStatusCalculator.class);

    private static final Set<GdSelfStatusEnum> MODERATION_SET = Set.of(DRAFT, RUN_WARN);

    private AdPrimaryStatusCalculator() {
    }

    public static GdiBannerPrimaryStatus convertToPrimaryStatus(@Nullable AggregatedStatusAdData statusData) {
        if (statusData == null || statusData.getStatus().isEmpty()) {
            return GdiBannerPrimaryStatus.DRAFT;
        }
        GdSelfStatusEnum status = statusData.getStatus().get();
        List<GdSelfStatusReason> reasons = ListUtils.emptyIfNull(statusData.getReasons());
        Collection<AdStatesEnum> states = emptyIfNull(statusData.getStates());

        // Архивные
        if (isArchivedPrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.ARCHIVED;
        }
        // На модерации
        else if (isModerationPrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.MODERATION;
        }
        // Отклоненные
        else if (isModerationRejectedPrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.MODERATION_REJECTED;
        }
        // Остановленные
        else if (isManuallySuspendedPrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.MANUALLY_SUSPENDED;
        }
        // Черновики
        else if (isDraftPrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.DRAFT;
        }
        // Запущенные
        else if (isActivePrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.ACTIVE;
        }
        // На данный момент не используется в фильтрах
        else if (isTemporarilySuspendedPrimaryStatus(status, reasons, states)) {
            return GdiBannerPrimaryStatus.TEMPORARILY_SUSPENDED;
        }

        logger.warn("Can't find primary status for {}", statusData);
        return GdiBannerPrimaryStatus.ACTIVE;
    }

    // Архивные
    private static boolean isArchivedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                   Collection<AdStatesEnum> states) {
        return Objects.equals(status, GdSelfStatusEnum.ARCHIVED);
    }

    // На модерации
    private static boolean isModerationPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                     Collection<AdStatesEnum> states) {
        return GdSelfStatusEnum.ON_MODERATION.equals(status)
                || (MODERATION_SET.contains(status) && reasons.contains(AD_ON_MODERATION))
                || (allStop().contains(status) && reasons.contains(ADGROUP_SHOW_CONDITIONS_ON_MODERATION));
    }

    // Отклоненные
    private static boolean isModerationRejectedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                             Collection<AdStatesEnum> states) {
        return (status.equals(STOP_CRIT)
                && containsAny(reasons, List.of(AD_VCARD_REJECTED_ON_MODERATION, REJECTED_ON_MODERATION)))
                || (
                status.equals(RUN_WARN)
                        && containsAny(reasons, List.of(
                        AD_VCARD_REJECTED_ON_MODERATION, AD_VIDEO_ADDITION_REJECTED_ON_MODERATION,
                        AD_IMAGE_REJECTED_ON_MODERATION, AD_DISPLAY_HREF_REJECTED_ON_MODERATION,
                        AD_TURBOLANDING_REJECTED_ON_MODERATION)))
                || (Set.of(DRAFT, STOP_CRIT).contains(status) && reasons.contains(ADGROUP_REJECTED_ON_MODERATION));
    }

    // Остановленные
    private static boolean isManuallySuspendedPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                            Collection<AdStatesEnum> states) {
        return GdSelfStatusEnum.allStop().contains(status) && status != GdSelfStatusEnum.STOP_PROCESSING
                || (
                status.equals(DRAFT)
                        && containsAny(List.of(
                        ADGROUP_SHOW_CONDITIONS_SUSPENDED_BY_USER, CAMPAIGN_SUSPENDED_BY_USER,
                        AD_SUSPENDED_BY_USER), reasons));
    }

    // Черновики
    private static boolean isDraftPrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                Collection<AdStatesEnum> states) {
        return status.equals(DRAFT);
    }

    // Запущенные
    private static boolean isActivePrimaryStatus(GdSelfStatusEnum status, List<GdSelfStatusReason> reasons,
                                                 Collection<AdStatesEnum> states) {
        return GdSelfStatusEnum.allRun().contains(status) || status == GdSelfStatusEnum.STOP_PROCESSING;
    }

    private static boolean isTemporarilySuspendedPrimaryStatus(GdSelfStatusEnum status,
                                                               List<GdSelfStatusReason> reasons,
                                                               Collection<AdStatesEnum> states) {
        return GdSelfStatusEnum.allPause().contains(status);
    }
}
